<?php
ob_start();
include 'include/header.php';
include '../config/conn.php';

// ✅ Ensure parent login
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'parent') {
    header('Location: login.php');
    exit();
}

$parent_id = $_SESSION['user_id'];
$error = '';

// ✅ Get children linked to this parent
$children_query = "
    SELECT s.id, s.full_name, s.student_id, s.class, s.section, 
           c.class_name, sec.section_name
    FROM students s
    LEFT JOIN classes c ON s.class = c.id
    LEFT JOIN sections sec ON s.section = sec.id
    WHERE s.parent_id = $parent_id
";
$children_result = mysqli_query($conn, $children_query);

// Check if query was successful
if (!$children_result) {
    $error = "Database error: " . mysqli_error($conn);
}

$children = [];
$child_ids = [];
if ($children_result && mysqli_num_rows($children_result) > 0) {
    while ($row = mysqli_fetch_assoc($children_result)) {
        $children[] = $row;
        $child_ids[] = $row['id'];
    }
}

// Initialize summary data
$summary_data = [
    'total_children' => count($children),
    'total_unpaid_invoices' => 0,
    'total_due_amount' => 0,
    'recent_notices' => 0,
    'upcoming_events' => 0
];

// Get financial summary if children exist
if (!empty($child_ids)) {
    $child_ids_str = implode(',', $child_ids);
    
    // Get unpaid invoices summary
    $invoices_query = "
        SELECT COUNT(*) as total_unpaid, COALESCE(SUM(i.amount - COALESCE(paid.paid_amount, 0)), 0) as total_due
        FROM invoices i
        LEFT JOIN (
            SELECT invoice_id, SUM(amount) as paid_amount 
            FROM payments 
            GROUP BY invoice_id
        ) paid ON i.id = paid.invoice_id
        WHERE i.student_id IN ($child_ids_str)
        AND (i.amount - COALESCE(paid.paid_amount, 0)) > 0
    ";
    
    $invoices_result = mysqli_query($conn, $invoices_query);
    if ($invoices_result) {
        $invoice_data = mysqli_fetch_assoc($invoices_result);
        $summary_data['total_unpaid_invoices'] = $invoice_data['total_unpaid'];
        $summary_data['total_due_amount'] = $invoice_data['total_due'];
    }
    
    // Get recent notices count (last 7 days)
    $notices_query = "
        SELECT COUNT(*) as recent_count
        FROM notices n
        WHERE (
            n.audience_type = 'all'
            OR (n.audience_type = 'class' AND n.target_id IN (SELECT class FROM students WHERE id IN ($child_ids_str)))
            OR (n.audience_type = 'section' AND n.target_id IN (SELECT section FROM students WHERE id IN ($child_ids_str)))
            OR (n.audience_type = 'student' AND n.target_id IN ($child_ids_str))
        )
        AND n.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        AND (n.expiry_date IS NULL OR n.expiry_date >= CURDATE())
    ";
    
    $notices_result = mysqli_query($conn, $notices_query);
    if ($notices_result) {
        $notice_data = mysqli_fetch_assoc($notices_result);
        $summary_data['recent_notices'] = $notice_data['recent_count'];
    }
    
    // Get upcoming events count (next 30 days)
    $events_query = "
        SELECT COUNT(*) as upcoming_count
        FROM events e
        WHERE e.event_date >= CURDATE() 
        AND e.event_date <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)
        AND (
            e.audience_type = 'all'
            OR (e.audience_type = 'class' AND e.target_id IN (SELECT class FROM students WHERE id IN ($child_ids_str)))
            OR (e.audience_type = 'section' AND e.target_id IN (SELECT section FROM students WHERE id IN ($child_ids_str)))
            OR (e.audience_type = 'student' AND e.target_id IN ($child_ids_str))
        )
    ";
    
    $events_result = mysqli_query($conn, $events_query);
    if ($events_result) {
        $event_data = mysqli_fetch_assoc($events_result);
        $summary_data['upcoming_events'] = $event_data['upcoming_count'];
    }
    
    // Get recent exam results for each child
    foreach ($children as &$child) {
        $child_id = $child['id'];
        
        // Get latest exam result
        $exam_query = "
            SELECT e.exam_name, er.subject_id, s.subject_name, er.ca_score, er.exam_score, 
                   (er.ca_score + er.exam_score) as total_score, er.grade, e.exam_date
            FROM exam_results er
            JOIN exams e ON er.exam_id = e.id
            JOIN subjects s ON er.subject_id = s.id
            WHERE er.student_id = $child_id 
            AND er.is_released = 1
            ORDER BY e.exam_date DESC, s.subject_name
            LIMIT 5
        ";
        
        $exam_result = mysqli_query($conn, $exam_query);
        $child['recent_results'] = [];
        if ($exam_result) {
            while ($exam_row = mysqli_fetch_assoc($exam_result)) {
                $child['recent_results'][] = $exam_row;
            }
        }
        
        // Get attendance summary
        $attendance_query = "
            SELECT 
                COUNT(*) as total_days,
                SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
                SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days,
                SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late_days
            FROM attendance 
            WHERE student_id = $child_id 
            AND attendance_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ";
        
        $attendance_result = mysqli_query($conn, $attendance_query);
        
        // Initialize with default values to prevent undefined key warnings
        $child['attendance'] = [
            'present_days' => 0, 
            'absent_days' => 0, 
            'late_days' => 0,
            'total_days' => 0,
            'attendance_rate' => 0
        ];
        
        if ($attendance_result && mysqli_num_rows($attendance_result) > 0) {
            $attendance_data = mysqli_fetch_assoc($attendance_result);
            $child['attendance'] = array_merge($child['attendance'], $attendance_data);
            
            if ($child['attendance']['total_days'] > 0) {
                $child['attendance']['attendance_rate'] = round(
                    ($child['attendance']['present_days'] / $child['attendance']['total_days']) * 100
                );
            }
        }
    }
    unset($child); // Unset reference
}

ob_end_flush();
?>

<div class="container mx-auto px-4 py-6">
    <!-- Welcome Header -->


    <?php if (!empty($error)): ?>
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
            <?php echo $error; ?>
        </div>
    <?php endif; ?>

    <!-- Summary Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div class="bg-white p-6 border-blue-500">
            <div class="flex items-center">
                <div class="bg-blue-100 p-3 r mr-4">
                    <i class="fas fa-user-graduate text-blue-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['total_children']; ?></h3>
                    <p class="text-gray-600">Children</p>
                </div>
            </div>
        </div>

        <div class="bg-white p-6 border-red-500">
            <div class="flex items-center">
                <div class="bg-red-100 p-3 r mr-4">
                    <i class="fas fa-file-invoice-dollar text-red-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['total_unpaid_invoices']; ?></h3>
                    <p class="text-gray-600">Unpaid Invoices</p>
                    <p class="text-sm text-red-600 font-semibold">
                        ₦<?php echo number_format($summary_data['total_due_amount'], 2); ?> due
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white p-6 border-green-500">
            <div class="flex items-center">
                <div class="bg-green-100 p-3 r mr-4">
                    <i class="fas fa-bell text-green-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['recent_notices']; ?></h3>
                    <p class="text-gray-600">Recent Notices</p>
                </div>
            </div>
        </div>

        <div class="bg-white p-6 border-purple-500">
            <div class="flex items-center">
                <div class="bg-purple-100 p-3 r mr-4">
                    <i class="fas fa-calendar-alt text-purple-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800"><?php echo $summary_data['upcoming_events']; ?></h3>
                    <p class="text-gray-600">Upcoming Events</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Children Overview -->
    <?php if (!empty($children)): ?>
        <div class="bg-white   mb-8">
            <div class="p-6 border-b">
                <h2 class="text-xl font-bold text-gray-800">Your Children</h2>
            </div>
            
            <div class="divide-y">
                <?php foreach ($children as $child): ?>
                    <div class="p-6">
                        <div class="flex justify-between items-start mb-4">
                            <div>
                                <h3 class="text-lg font-semibold text-gray-800"><?php echo htmlspecialchars($child['full_name']); ?></h3>
                                <p class="text-gray-600">
                                    <?php echo htmlspecialchars($child['class_name'] . ' - ' . $child['section_name']); ?> |
                                    ID: <?php echo htmlspecialchars($child['student_id']); ?>
                                </p>
                            </div>
                            <div class="text-right">
                                <span class="px-3 py-1 r text-xs font-medium 
                                    <?php echo $child['attendance']['attendance_rate'] >= 90 ? 'bg-green-100 text-green-800' : 
                                           ($child['attendance']['attendance_rate'] >= 75 ? 'bg-yellow-100 text-yellow-800' : 
                                           'bg-red-100 text-red-800'); ?>">
                                    Attendance: <?php echo $child['attendance']['attendance_rate']; ?>%
                                </span>
                            </div>
                        </div>

                        <!-- Recent Results -->
                        <?php if (!empty($child['recent_results'])): ?>
                            <div class="mb-4">
                                <h4 class="font-medium text-gray-700 mb-2">Recent Exam Results:</h4>
                                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-2">
                                    <?php foreach ($child['recent_results'] as $result): ?>
                                        <div class="bg-gray-50 p-3 rounded text-center">
                                            <div class="text-sm font-medium text-gray-600"><?php echo htmlspecialchars($result['subject_name']); ?></div>
                                            <div class="text-lg font-bold 
                                                <?php echo $result['grade'] === 'A' || $result['grade'] === 'A+' ? 'text-green-600' : 
                                                       ($result['grade'] === 'B' || $result['grade'] === 'B+' ? 'text-blue-600' : 
                                                       ($result['grade'] === 'C' ? 'text-yellow-600' : 'text-red-600')); ?>">
                                                <?php echo $result['grade']; ?>
                                            </div>
                                            <div class="text-xs text-gray-500"><?php echo $result['total_score']; ?> pts</div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php else: ?>
                            <p class="text-gray-500 text-sm mb-4">No recent exam results available.</p>
                        <?php endif; ?>

                        <!-- Quick Actions -->
                        <div class="flex space-x-3">
                            <a href="result.php?student_id=<?php echo $child['id']; ?>" 
                               class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 rounded text-sm flex items-center">
                                <i class="fas fa-chart-line mr-1"></i> View Results
                            </a>
                           
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php else: ?>
        <div class="bg-white p-8   text-center">
            <i class="fas fa-user-graduate text-4xl text-gray-300 mb-4"></i>
            <h3 class="text-lg font-medium text-gray-600">No children linked to your account</h3>
            <p class="text-gray-500">Please contact the school administration to link your children to your parent account.</p>
        </div>
    <?php endif; ?>

    <!-- Quick Links -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        <a href="notices.php" class="bg-white p-6   hover: transition-">
            <div class="flex items-center">
                <div class="bg-blue-100 p-3 r mr-4">
                    <i class="fas fa-bell text-blue-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="font-semibold text-gray-800">View Notices</h3>
                    <p class="text-sm text-gray-600">Check school announcements</p>
                </div>
            </div>
        </a>

        <a href="events.php" class="bg-white p-6   hover: transition-">
            <div class="flex items-center">
                <div class="bg-green-100 p-3 r mr-4">
                    <i class="fas fa-calendar-alt text-green-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="font-semibold text-gray-800">School Events</h3>
                    <p class="text-sm text-gray-600">View upcoming events</p>
                </div>
            </div>
        </a>

        <a href="profile.php" class="bg-white p-6   hover: transition-">
            <div class="flex items-center">
                <div class="bg-purple-100 p-3 r mr-4">
                    <i class="fas fa-user-cog text-purple-600 text-xl"></i>
                </div>
                <div>
                    <h3 class="font-semibold text-gray-800">Profile Settings</h3>
                    <p class="text-sm text-gray-600">Update your information</p>
                </div>
            </div>
        </a>
    </div>
</div>

<?php
include 'include/footer.php';
?>