<?php
// ajax_load_students.php
session_start();
include '../config/conn.php';

// Check if logged in
if (!isset($_SESSION['user_id'])) {
    header('HTTP/1.1 401 Unauthorized');
    echo json_encode(['error' => 'Unauthorized access']);
    exit();
}

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('HTTP/1.1 405 Method Not Allowed');
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

// Get and validate parameters
$class_id = isset($_POST['class_id']) ? intval($_POST['class_id']) : 0;
$section_id = isset($_POST['section_id']) ? intval($_POST['section_id']) : 0;
$attendance_date = isset($_POST['attendance_date']) ? $_POST['attendance_date'] : date('Y-m-d');

if ($class_id <= 0 || $section_id <= 0 || empty($attendance_date)) {
    header('HTTP/1.1 400 Bad Request');
    echo json_encode(['error' => 'Invalid parameters']);
    exit();
}

// Fetch class and section names
$class_query = "SELECT class_name FROM classes WHERE id = $class_id";
$class_result = mysqli_query($conn, $class_query);
$class_name = mysqli_fetch_assoc($class_result)['class_name'] ?? 'Unknown Class';

$section_query = "SELECT section_name FROM sections WHERE id = $section_id";
$section_result = mysqli_query($conn, $section_query);
$section_name = mysqli_fetch_assoc($section_result)['section_name'] ?? 'Unknown Section';

// Fetch students based on selected class and section
$students_query = "SELECT id, student_id, full_name, roll_number, email 
                   FROM students 
                   WHERE class = $class_id AND section = $section_id AND status = 'active' 
                   ORDER BY roll_number";
$students_result = mysqli_query($conn, $students_query);
$students = [];
while ($student = mysqli_fetch_assoc($students_result)) {
    $students[] = $student;
}

// Check existing attendance
$existing_attendance = [];
$attendance_exists = false;
$check_attendance_query = "SELECT student_id, status, recorded_at 
                           FROM attendance 
                           WHERE class_id = $class_id 
                           AND section_id = $section_id 
                           AND attendance_date = '$attendance_date'";
$check_attendance_result = mysqli_query($conn, $check_attendance_query);
if (mysqli_num_rows($check_attendance_result) > 0) {
    $attendance_exists = true;
    while ($row = mysqli_fetch_assoc($check_attendance_result)) {
        $existing_attendance[$row['student_id']] = $row;
    }
}

// Generate HTML response
ob_start();
?>
<!-- Attendance Form -->
<div class="mb-6 bg-white p-4 rounded-lg shadow">
    <div class="flex justify-between items-center mb-4">
        <h3 class="text-lg font-semibold text-gray-800">
            Mark Student Attendance for <?php 
            echo $class_name . " - " . $section_name . " (" . date('d M Y', strtotime($attendance_date)) . ")";
            ?>
        </h3>
        
        <?php if ($attendance_exists): ?>
            <span class="bg-blue-100 text-blue-800 text-xs font-medium px-2.5 py-0.5 rounded-full">
                <i class="fas fa-info-circle mr-1"></i> Attendance already recorded for this date
            </span>
        <?php endif; ?>
    </div>
    
    <?php if (!empty($students)): ?>
        <form id="attendance-form" method="POST" action="">
            <input type="hidden" name="class_id" value="<?php echo $class_id; ?>">
            <input type="hidden" name="section_id" value="<?php echo $section_id; ?>">
            <input type="hidden" name="attendance_date" value="<?php echo $attendance_date; ?>">
            
            <div class="overflow-x-auto">
                <table class="min-w-full bg-white">
                    <thead class="bg-gray-200 text-gray-700">
                        <tr>
                            <th class="py-3 px-4 text-left">Roll No</th>
                            <th class="py-3 px-4 text-left">Student ID</th>
                            <th class="py-3 px-4 text-left">Name</th>
                            <th class="py-3 px-4 text-left">Status</th>
                            <th class="py-3 px-4 text-left">Time</th>
                        </tr>
                    </thead>
                    <tbody class="text-gray-600">
                        <?php foreach ($students as $student): 
                            $status = isset($existing_attendance[$student['id']]) ? $existing_attendance[$student['id']]['status'] : '';
                            $time = isset($existing_attendance[$student['id']]) ? date('H:i', strtotime($existing_attendance[$student['id']]['recorded_at'])) : '';
                        ?>
                            <tr class="border-b border-gray-200 hover:bg-gray-50">
                                <td class="py-3 px-4"><?php echo $student['roll_number']; ?></td>
                                <td class="py-3 px-4"><?php echo $student['student_id']; ?></td>
                                <td class="py-3 px-4"><?php echo $student['full_name']; ?></td>
                                <td class="py-3 px-4">
                                    <div class="flex flex-wrap gap-4">
                                        <label class="inline-flex items-center">
                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="present" 
                                                <?php echo ($status == 'present') ? 'checked' : 'checked'; ?> 
                                                class="form-radio h-4 w-4 text-green-600">
                                            <span class="ml-2 text-green-600">Present</span>
                                        </label>
                                        <label class="inline-flex items-center">
                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="absent" 
                                                <?php echo ($status == 'absent') ? 'checked' : ''; ?> 
                                                class="form-radio h-4 w-4 text-red-600">
                                            <span class="ml-2 text-red-600">Absent</span>
                                        </label>
                                        <label class="inline-flex items-center">
                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="late" 
                                                <?php echo ($status == 'late') ? 'checked' : ''; ?> 
                                                class="form-radio h-4 w-4 text-yellow-600">
                                            <span class="ml-2 text-yellow-600">Late</span>
                                        </label>
                                        <label class="inline-flex items-center">
                                            <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="half_day" 
                                                <?php echo ($status == 'half_day') ? 'checked' : ''; ?> 
                                                class="form-radio h-4 w-4 text-purple-600">
                                            <span class="ml-2 text-purple-600">Half Day</span>
                                        </label>
                                    </div>
                                </td>
                                <td class="py-3 px-4">
                                    <?php if (!empty($time)): ?>
                                        <span class="attendance-status status-<?php echo $status; ?>">
                                            <?php echo $time; ?>
                                        </span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <div class="mt-6 flex justify-center space-x-4">
                <button type="button" onclick="selectAllPresent()" class="bg-green-500 hover:bg-green-600 text-white px-4 py-2 rounded-lg">
                    <i class="fas fa-check-circle mr-2"></i> Mark All Present
                </button>
                <button type="button" onclick="selectAllAbsent()" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg">
                    <i class="fas fa-times-circle mr-2"></i> Mark All Absent
                </button>
                <button type="submit" id="save-attendance-btn" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg flex items-center">
                    <i class="fas fa-save mr-2"></i> Save Attendance
                </button>
            </div>
        </form>
    <?php else: ?>
        <p class="text-center py-4 text-gray-500">
            No active students found in this class and section.
        </p>
    <?php endif; ?>
</div>

<!-- Calendar View -->
<?php
// Calendar variables
$calendar_month = date('Y-m');
$month_name = date('F Y', strtotime($calendar_month));
$first_day = date('N', strtotime($calendar_month . '-01'));
$days_in_month = date('t', strtotime($calendar_month . '-01'));

// Get calendar attendance data
$calendar_data = [];
$calendar_query = "SELECT attendance_date, 
                   SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present,
                   SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent,
                   SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late,
                   SUM(CASE WHEN status = 'half_day' THEN 1 ELSE 0 END) as half_day
                   FROM attendance 
                   WHERE class_id = $class_id 
                   AND section_id = $section_id 
                   AND DATE_FORMAT(attendance_date, '%Y-%m') = '$calendar_month'
                   GROUP BY attendance_date";
$calendar_result = mysqli_query($conn, $calendar_query);
while ($row = mysqli_fetch_assoc($calendar_result)) {
    $calendar_data[$row['attendance_date']] = $row;
}
?>
<div class="mt-6 bg-white p-4 rounded-lg shadow">
    <h3 class="text-lg font-semibold text-gray-800 mb-4">Attendance Calendar</h3>
    
    <div class="mb-4 flex justify-between items-center">
        <h4 class="text-md font-medium text-gray-700"><?php echo $month_name; ?></h4>
        
        <form method="GET" action="" class="flex items-center">
            <input type="hidden" name="class_id" value="<?php echo $class_id; ?>">
            <input type="hidden" name="section_id" value="<?php echo $section_id; ?>">
            <input type="hidden" name="attendance_date" value="<?php echo $attendance_date; ?>">
            <input type="month" name="calendar_month" value="<?php echo $calendar_month; ?>" 
                class="px-3 py-1 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 mr-2">
            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 rounded-lg">
                View
            </button>
        </form>
    </div>
    
    <div class="grid grid-cols-7 gap-2 mb-2">
        <div class="text-center font-medium text-gray-700">Mon</div>
        <div class="text-center font-medium text-gray-700">Tue</div>
        <div class="text-center font-medium text-gray-700">Wed</div>
        <div class="text-center font-medium text-gray-700">Thu</div>
        <div class="text-center font-medium text-gray-700">Fri</div>
        <div class="text-center font-medium text-gray-700">Sat</div>
        <div class="text-center font-medium text-gray-700">Sun</div>
    </div>
    
    <div class="grid grid-cols-7 gap-2">
        <?php
        // Add empty cells for days before the first day of the month
        for ($i = 1; $i < $first_day; $i++) {
            echo '<div class="calendar-day bg-gray-100"></div>';
        }
        
        // Add cells for each day of the month
        for ($day = 1; $day <= $days_in_month; $day++) {
            $current_date = date('Y-m-d', strtotime($calendar_month . '-' . sprintf('%02d', $day)));
            $is_today = ($current_date == date('Y-m-d')) ? 'today' : '';
            $has_attendance = isset($calendar_data[$current_date]);
            
            echo '<div class="calendar-day ' . $is_today . ' ' . ($has_attendance ? 'has-attendance' : '') . '">';
            echo '<div class="text-sm font-medium">' . $day . '</div>';
            
            if ($has_attendance) {
                $data = $calendar_data[$current_date];
                echo '<div class="attendance-stats">';
                echo '<div class="text-green-600">P: ' . $data['present'] . '</div>';
                echo '<div class="text-red-600">A: ' . $data['absent'] . '</div>';
                echo '<div class="text-yellow-600">L: ' . $data['late'] . '</div>';
                echo '<div class="text-purple-600">H: ' . $data['half_day'] . '</div>';
                echo '</div>';
            }
            
            echo '</div>';
            
            // Start new row after Sunday
            if (($first_day + $day - 1) % 7 == 0 && $day != $days_in_month) {
                echo '</div><div class="grid grid-cols-7 gap-2">';
            }
        }
        
        // Add empty cells for remaining days in the last week
        $last_day = date('N', strtotime($calendar_month . '-' . $days_in_month));
        if ($last_day < 7) {
            for ($i = $last_day; $i < 7; $i++) {
                echo '<div class="calendar-day bg-gray-100"></div>';
            }
        }
        ?>
    </div>
    
    <div class="mt-4 flex justify-center items-center space-x-4 text-xs">
        <div class="flex items-center">
            <div class="w-4 h-4 bg-blue-100 mr-1 border border-blue-300"></div>
            <span>Today</span>
        </div>
        <div class="flex items-center">
            <div class="w-4 h-4 bg-blue-50 mr-1"></div>
            <span>Has Attendance</span>
        </div>
    </div>
</div>

<script>
// Reattach event handlers for the newly loaded content
$(document).ready(function() {
    // AJAX for attendance submission
    $('#attendance-form').on('submit', function(e) {
        e.preventDefault();
        
        // Show loading state
        const btn = $('#save-attendance-btn');
        const originalText = btn.html();
        btn.html('<span class="loading mr-2"></span> Saving...').prop('disabled', true);
        
        // Hide any previous messages
        $('#success-message, #error-message').addClass('hidden');
        
        // Submit form via AJAX
        $.ajax({
            url: 'ajax_save_attendance.php',
            type: 'POST',
            data: $(this).serialize(),
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    // Show success message
                    $('#success-message').text(response.message).removeClass('hidden');
                    
                    // Update any existing attendance times if needed
                    if (response.updated_records) {
                        // You could update the UI here if needed
                        location.reload(); // Or update specific elements instead of reloading
                    }
                } else {
                    // Show error message
                    $('#error-message').text(response.message).removeClass('hidden');
                }
            },
            error: function() {
                $('#error-message').text('An error occurred while saving attendance.').removeClass('hidden');
            },
            complete: function() {
                // Restore button
                btn.html(originalText).prop('disabled', false);
            }
        });
    });
});
</script>
<?php
$html = ob_get_clean();

// Return the HTML response
echo $html;

mysqli_close($conn);
?>