<?php 
include 'include/header.php';
include '../config/conn.php';

// Initialize all variables with default values
$total_students = 0;
$total_teachers = 0;
$total_revenue = 0;
$total_subjects = 0;
$attendance_percentage = 0;
$teacher_attendance_percentage = 0;
$paid_invoices = [];
$recent_payments = [];
$upcoming_events = [];
$recent_notices = [];
$monthly_revenue_data = [];
$monthly_labels = [];
$class_distribution = [];
$weekly_attendance_labels = [];
$weekly_attendance_present = [];
$weekly_attendance_absent = [];
$weekly_teacher_attendance_labels = [];
$weekly_teacher_attendance_present = [];
$weekly_teacher_attendance_absent = [];

// Get total students
$students_query = "SELECT COUNT(*) as total FROM students WHERE status = 'active'";
if ($students_result = mysqli_query($conn, $students_query)) {
    if ($students_data = mysqli_fetch_assoc($students_result)) {
        $total_students = $students_data['total'];
    }
}

// Get total teachers
$teachers_query = "SELECT COUNT(*) as total FROM teachers WHERE status = 'active'";
if ($teachers_result = mysqli_query($conn, $teachers_query)) {
    if ($teachers_data = mysqli_fetch_assoc($teachers_result)) {
        $total_teachers = $teachers_data['total'];
    }
}

// Get total revenue (sum of all payments)
$revenue_query = "SELECT COALESCE(SUM(amount), 0) as total FROM payments";
if ($revenue_result = mysqli_query($conn, $revenue_query)) {
    if ($revenue_data = mysqli_fetch_assoc($revenue_result)) {
        $total_revenue = $revenue_data['total'];
    }
}

// Get total subjects
$subjects_query = "SELECT COUNT(*) as total FROM subjects WHERE status = 'active'";
if ($subjects_result = mysqli_query($conn, $subjects_query)) {
    if ($subjects_data = mysqli_fetch_assoc($subjects_result)) {
        $total_subjects = $subjects_data['total'];
    }
}

// Get today's student attendance
$today = date('Y-m-d');
$attendance_query = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present,
                    SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent
                    FROM attendance WHERE attendance_date = '$today'";
if ($attendance_result = mysqli_query($conn, $attendance_query)) {
    if ($attendance_data = mysqli_fetch_assoc($attendance_result)) {
        $attendance_percentage = $attendance_data['total'] > 0 ? 
                                round(($attendance_data['present'] / $attendance_data['total']) * 100, 2) : 0;
    }
}

// Get today's teacher attendance
$teacher_attendance_query = "SELECT 
                            COUNT(*) as total,
                            SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present,
                            SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent
                            FROM teacher_attendance WHERE attendance_date = '$today'";
if ($teacher_attendance_result = mysqli_query($conn, $teacher_attendance_query)) {
    if ($teacher_attendance_data = mysqli_fetch_assoc($teacher_attendance_result)) {
        $teacher_attendance_percentage = $teacher_attendance_data['total'] > 0 ? 
                                        round(($teacher_attendance_data['present'] / $teacher_attendance_data['total']) * 100, 2) : 0;
    }
}

// Get recent paid invoices
$paid_invoices_query = "SELECT i.*, s.full_name as student_name, 
                       (SELECT COALESCE(SUM(amount), 0) FROM payments p WHERE p.invoice_id = i.id) as paid_amount
                       FROM invoices i
                       JOIN students s ON i.student_id = s.id
                       WHERE i.status = 'paid'
                       ORDER BY i.created_at DESC 
                       LIMIT 5";
if ($paid_invoices_result = mysqli_query($conn, $paid_invoices_query)) {
    while ($invoice = mysqli_fetch_assoc($paid_invoices_result)) {
        $paid_invoices[] = $invoice;
    }
}

// Get recent payments
$recent_payments_query = "SELECT p.*, s.full_name as student_name 
                         FROM payments p 
                         JOIN students s ON p.student_id = s.id 
                         ORDER BY p.payment_date DESC 
                         LIMIT 5";
if ($recent_payments_result = mysqli_query($conn, $recent_payments_query)) {
    while ($payment = mysqli_fetch_assoc($recent_payments_result)) {
        $recent_payments[] = $payment;
    }
}

// Get upcoming events (assuming you have an events table)
$events_query = "SELECT * FROM events 
                WHERE event_date >= CURDATE() 
                ORDER BY event_date ASC 
                LIMIT 3";
if ($events_result = mysqli_query($conn, $events_query)) {
    while ($event = mysqli_fetch_assoc($events_result)) {
        $upcoming_events[] = $event;
    }
}

// Get recent notices
$notices_query = "SELECT * FROM notices 
                 ORDER BY created_at DESC 
                 LIMIT 3";
if ($notices_result = mysqli_query($conn, $notices_query)) {
    while ($notice = mysqli_fetch_assoc($notices_result)) {
        $recent_notices[] = $notice;
    }
}

// Get monthly revenue data for chart
$monthly_revenue_query = "SELECT 
                         YEAR(payment_date) as year, 
                         MONTH(payment_date) as month, 
                         SUM(amount) as total 
                         FROM payments 
                         WHERE payment_date >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
                         GROUP BY YEAR(payment_date), MONTH(payment_date)
                         ORDER BY year, month";
if ($monthly_revenue_result = mysqli_query($conn, $monthly_revenue_query)) {
    while ($row = mysqli_fetch_assoc($monthly_revenue_result)) {
        $monthly_revenue_data[] = $row['total'];
        $monthly_labels[] = date('M Y', mktime(0, 0, 0, $row['month'], 1, $row['year']));
    }
}

// Get class-wise student distribution
$class_distribution_query = "SELECT c.class_name, COUNT(s.id) as student_count 
                            FROM students s 
                            JOIN classes c ON s.class = c.id 
                            WHERE s.status = 'active' 
                            GROUP BY s.class 
                            ORDER BY student_count DESC 
                            LIMIT 5";
if ($class_distribution_result = mysqli_query($conn, $class_distribution_query)) {
    while ($row = mysqli_fetch_assoc($class_distribution_result)) {
        $class_distribution[] = $row;
    }
}

// Get weekly attendance data for students
$weekly_attendance_query = "SELECT 
                           attendance_date,
                           SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present,
                           SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent
                           FROM attendance 
                           WHERE attendance_date >= DATE_SUB('$today', INTERVAL 7 DAY)
                           GROUP BY attendance_date
                           ORDER BY attendance_date";
if ($weekly_attendance_result = mysqli_query($conn, $weekly_attendance_query)) {
    while ($row = mysqli_fetch_assoc($weekly_attendance_result)) {
        $weekly_attendance_labels[] = date('D, M j', strtotime($row['attendance_date']));
        $weekly_attendance_present[] = $row['present'];
        $weekly_attendance_absent[] = $row['absent'];
    }
}

// Get weekly attendance data for teachers
$weekly_teacher_attendance_query = "SELECT 
                                   attendance_date,
                                   SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present,
                                   SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent
                                   FROM teacher_attendance 
                                   WHERE attendance_date >= DATE_SUB('$today', INTERVAL 7 DAY)
                                   GROUP BY attendance_date
                                   ORDER BY attendance_date";
if ($weekly_teacher_attendance_result = mysqli_query($conn, $weekly_teacher_attendance_query)) {
    while ($row = mysqli_fetch_assoc($weekly_teacher_attendance_result)) {
        $weekly_teacher_attendance_labels[] = date('D, M j', strtotime($row['attendance_date']));
        $weekly_teacher_attendance_present[] = $row['present'];
        $weekly_teacher_attendance_absent[] = $row['absent'];
    }
}
?>

<!-- Main content -->
<div class="flex-1 flex flex-col overflow-hidden">
    <!-- Main content -->
    <main class="flex-1 overflow-y-auto bg-gray-50 p-6">
        <!-- Welcome Header -->
        <div class="mb-8">
            <h1 class="text-2xl md:text-3xl font-bold text-gray-800">Admin Dashboard</h1>
            <p class="text-gray-600 mt-1">Welcome back, <?php echo $_SESSION['user_name'] ?? 'Administrator'; ?>!</p>
        </div>
        
        <!-- Stats -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <div class="bg-white   p-6 dashboard-card transition-all duration-300 hover:shadow-lg">
                <div class="flex items-center">
                    <div class="p-3  bg-blue-100 text-blue-600">
                        <i class="fas fa-user-graduate text-2xl"></i>
                    </div>
                    <div class="ml-4">
                        <h3 class="text-lg font-semibold text-gray-600">Total Students</h3>
                        <p class="text-xl font-bold text-gray-800"><?php echo number_format($total_students); ?></p>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-100 flex items-center text-sm text-green-600">
                    <i class="fas fa-users mr-1"></i>
                    <span><?php echo number_format($attendance_percentage); ?>% attendance today</span>
                </div>
            </div>
            
            <div class="bg-white   p-6 dashboard-card transition-all duration-300 hover:shadow-lg">
                <div class="flex items-center">
                    <div class="p-3  bg-green-100 text-green-600">
                        <i class="fas fa-chalkboard-teacher text-2xl"></i>
                    </div>
                    <div class="ml-4">
                        <h3 class="text-lg font-semibold text-gray-600">Teaching Staff</h3>
                        <p class="text-xl font-bold text-gray-800"><?php echo number_format($total_teachers); ?></p>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-100 flex items-center text-sm text-green-600">
                    <i class="fas fa-user-check mr-1"></i>
                    <span><?php echo number_format($teacher_attendance_percentage); ?>% attendance today</span>
                </div>
            </div>
            
            <div class="bg-white   p-6 dashboard-card transition-all duration-300 hover:shadow-lg">
                <div class="flex items-center">
                    <div class="p-3  bg-red-100 text-red-600">
                        <i class="fas fa-money-bill-wave text-2xl"></i>
                    </div>
                    <div class="ml-4">
                        <h3 class="text font-semibold text-gray-600">Total Revenue</h3>
                        <p class="text-xl font-bold text-gray-800">₦<?php echo number_format($total_revenue, 2); ?></p>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-100 flex items-center text-sm text-green-600">
                
                </div>
            </div>
            
            <div class="bg-white   p-6 dashboard-card transition-all duration-300 hover:shadow-lg">
                <div class="flex items-center">
                    <div class="p-3  bg-yellow-100 text-yellow-600">
                        <i class="fas fa-book-open text-2xl"></i>
                    </div>
                    <div class="ml-4">
                        <h3 class="text-lg font-semibold text-gray-600">Active Subjects</h3>
                        <p class="text-xl font-bold text-gray-800"><?php echo number_format($total_subjects); ?></p>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-100 flex items-center text-sm text-green-600">

                </div>
            </div>
        </div>
        
        <!-- Charts and recent activities -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
            <div class="bg-white   p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Revenue Trend (Last 6 Months)</h3>
                <div class="h-80">
                    <canvas id="revenueChart"></canvas>
                </div>
            </div>
            
            <div class="bg-white   p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Recent Payments</h3>
                <div class="space-y-4">
                    <?php if (!empty($recent_payments)): ?>
                        <?php foreach ($recent_payments as $payment): ?>
                            <div class="flex items-start p-3 bg-gray-50  hover:bg-blue-50 transition-colors">
                                <div class="p-2  bg-green-100 text-green-600 mr-3">
                                    <i class="fas fa-money-check"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-medium text-gray-800">Payment received from <?php echo $payment['student_name']; ?></p>
                                    <p class="text-xs text-gray-600">₦<?php echo number_format($payment['amount'], 2); ?> via <?php echo ucfirst($payment['payment_method']); ?></p>
                                    <p class="text-xs text-gray-400 mt-1"><?php echo date('M j, Y g:i A', strtotime($payment['payment_date'])); ?></p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p class="text-gray-500 text-center py-4">No recent payments found</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Student and Teacher Attendance Charts -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
            <div class="bg-white   p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Student Attendance (Last 7 Days)</h3>
                <div class="h-80">
                    <canvas id="studentAttendanceChart"></canvas>
                </div>
            </div>
            
            <div class="bg-white   p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Teacher Attendance (Last 7 Days)</h3>
                <div class="h-80">
                    <canvas id="teacherAttendanceChart"></canvas>
                </div>
            </div>
        </div>
        
        <!-- Class distribution and paid invoices -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
            <div class="bg-white   p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Class Distribution</h3>
                <div class="h-80">
                    <canvas id="classDistributionChart"></canvas>
                </div>
            </div>
            
            <div class="bg-white   p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Recently Paid Invoices</h3>
                    <a href="view_invoices.php" class="text-sm text-blue-600 hover:text-blue-800 flex items-center">
                        View All <i class="fas fa-arrow-right ml-1 text-xs"></i>
                    </a>
                </div>
                <div class="space-y-4">
                    <?php if (!empty($paid_invoices)): ?>
                        <?php foreach ($paid_invoices as $invoice): ?>
                            <div class="flex items-start p-3 bg-green-50  hover:bg-green-100 transition-colors">
                                <div class="p-2  bg-green-100 text-green-600 mr-3">
                                    <i class="fas fa-receipt"></i>
                                </div>
                                <div class="flex-1">
                                    <p class="font-medium text-gray-800"><?php echo $invoice['student_name']; ?></p>
                                    <p class="text-sm text-gray-600"><?php echo $invoice['description']; ?></p>
                                    <div class="flex justify-between items-center mt-2">
                                        <p class="text-xs text-gray-500">₦<?php echo number_format($invoice['amount'], 2); ?></p>
                                        <span class="px-2 py-1 bg-green-100 text-green-800 text-xs ">Paid</span>
                                    </div>
                                    <p class="text-xs text-gray-400 mt-1">Paid on: <?php echo date('M j, Y', strtotime($invoice['created_at'])); ?></p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p class="text-gray-500 text-center py-4">No paid invoices found</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Upcoming events and notices -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div class="bg-white   p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Upcoming Events</h3>
                    <a href="events.php" class="text-sm text-blue-600 hover:text-blue-800 flex items-center">
                        View All <i class="fas fa-arrow-right ml-1 text-xs"></i>
                    </a>
                </div>
                <div class="space-y-4">
                    <?php if (!empty($upcoming_events)): ?>
                        <?php foreach ($upcoming_events as $event): ?>
                            <div class="flex items-center p-3 bg-blue-50  hover:bg-blue-100 transition-colors">
                                <div class="bg-blue-600 text-white text-center p-2  mr-3 w-16">
                                    <p class="text-sm font-bold"><?php echo date('d', strtotime($event['event_date'])); ?></p>
                                    <p class="text-xs"><?php echo strtoupper(date('M', strtotime($event['event_date']))); ?></p>
                                </div>
                                <div>
                                    <p class="font-medium text-gray-800"><?php echo $event['title']; ?></p>
                                    <p class="text-xs text-gray-600">
                                        <?php echo date('g:i A', strtotime($event['start_time'])); ?> - 
                                        <?php echo date('g:i A', strtotime($event['end_time'])); ?>
                                    </p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p class="text-gray-500 text-center py-4">No upcoming events</p>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="bg-white   p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Notices</h3>
                    <a href="notices.php" class="text-sm text-blue-600 hover:text-blue-800 flex items-center">
                        View All <i class="fas fa-arrow-right ml-1 text-xs"></i>
                    </a>
                </div>
                <div class="space-y-4">
                    <?php if (!empty($recent_notices)): ?>
                        <?php foreach ($recent_notices as $notice): 
                            $border_color = 'border-blue-500';
                            $bg_color = 'bg-blue-50';
                            
                            if ($notice['priority'] == 'high') {
                                $border_color = 'border-red-500';
                                $bg_color = 'bg-red-50';
                            } elseif ($notice['priority'] == 'medium') {
                                $border_color = 'border-yellow-500';
                                $bg_color = 'bg-yellow-50';
                            }
                        ?>
                            <div class="p-3 border-l-4 <?php echo $border_color; ?> <?php echo $bg_color; ?> rounded-r-lg hover:shadow-sm transition-shadow">
                                <p class="font-medium text-gray-800"><?php echo $notice['title']; ?></p>
                                <p class="text-sm text-gray-600"><?php echo substr($notice['content'], 0, 100); ?>...</p>
                                <p class="text-xs text-gray-400 mt-1">Posted on: <?php echo date('M j, Y', strtotime($notice['created_at'])); ?></p>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <p class="text-gray-500 text-center py-4">No notices found</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>
</div>

<!-- Include Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels"></script>

<script>
// Revenue Chart (3D Line Chart)
const revenueCtx = document.getElementById('revenueChart').getContext('2d');
const revenueChart = new Chart(revenueCtx, {
    type: 'line',
    data: {
        labels: <?php echo json_encode($monthly_labels); ?>,
        datasets: [{
            label: 'Monthly Revenue (₦)',
            data: <?php echo json_encode($monthly_revenue_data); ?>,
            backgroundColor: 'rgba(54, 162, 235, 0.2)',
            borderColor: 'rgba(54, 162, 235, 1)',
            borderWidth: 3,
            tension: 0.4,
            fill: true,
            pointBackgroundColor: 'rgba(54, 162, 235, 1)',
            pointBorderColor: '#fff',
            pointBorderWidth: 2,
            pointRadius: 5,
            pointHoverRadius: 7
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                display: true,
                position: 'top',
            },
            tooltip: {
                mode: 'index',
                intersect: false,
                callbacks: {
                    label: function(context) {
                        return '₦' + context.parsed.y.toLocaleString();
                    }
                }
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    callback: function(value) {
                        return '₦' + value.toLocaleString();
                    }
                },
                grid: {
                    color: 'rgba(0, 0, 0, 0.1)'
                }
            },
            x: {
                grid: {
                    color: 'rgba(0, 0, 0, 0.1)'
                }
            }
        }
    }
});

// Class Distribution Chart (3D Doughnut Chart)
const classCtx = document.getElementById('classDistributionChart').getContext('2d');
const classChart = new Chart(classCtx, {
    type: 'doughnut',
    data: {
        labels: <?php echo json_encode(array_column($class_distribution, 'class_name')); ?>,
        datasets: [{
            data: <?php echo json_encode(array_column($class_distribution, 'student_count')); ?>,
            backgroundColor: [
                'rgba(255, 99, 132, 0.8)',
                'rgba(54, 162, 235, 0.8)',
                'rgba(255, 206, 86, 0.8)',
                'rgba(75, 192, 192, 0.8)',
                'rgba(153, 102, 255, 0.8)'
            ],
            borderColor: [
                'rgba(255, 99, 132, 1)',
                'rgba(54, 162, 235, 1)',
                'rgba(255, 206, 86, 1)',
                'rgba(75, 192, 192, 1)',
                'rgba(153, 102, 255, 1)'
            ],
            borderWidth: 2,
            hoverOffset: 15
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'right',
            },
            tooltip: {
                callbacks: {
                    label: function(context) {
                        return context.label + ': ' + context.raw + ' students';
                    }
                }
            }
        },
        cutout: '60%'
    }
});

// Student Attendance Chart (2D Bar Chart)
const studentAttendanceCtx = document.getElementById('studentAttendanceChart').getContext('2d');
const studentAttendanceChart = new Chart(studentAttendanceCtx, {
    type: 'bar',
    data: {
        labels: <?php echo json_encode($weekly_attendance_labels); ?>,
        datasets: [
            {
                label: 'Present',
                data: <?php echo json_encode($weekly_attendance_present); ?>,
                backgroundColor: 'rgba(75, 192, 192, 0.7)',
                borderColor: 'rgba(75, 192, 192, 1)',
                borderWidth: 1
            },
            {
                label: 'Absent',
                data: <?php echo json_encode($weekly_attendance_absent); ?>,
                backgroundColor: 'rgba(255, 99, 132, 0.7)',
                borderColor: 'rgba(255, 99, 132, 1)',
                borderWidth: 1
            }
        ]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
            y: {
                beginAtZero: true,
                title: {
                    display: true,
                    text: 'Number of Students'
                },
                ticks: {
                    stepSize: 5
                }
            },
            x: {
                title: {
                    display: true,
                    text: 'Date'
                }
            }
        }
    }
});

// Teacher Attendance Chart (2D Bar Chart)
const teacherAttendanceCtx = document.getElementById('teacherAttendanceChart').getContext('2d');
const teacherAttendanceChart = new Chart(teacherAttendanceCtx, {
    type: 'bar',
    data: {
        labels: <?php echo json_encode($weekly_teacher_attendance_labels); ?>,
        datasets: [
            {
                label: 'Present',
                data: <?php echo json_encode($weekly_teacher_attendance_present); ?>,
                backgroundColor: 'rgba(54, 162, 235, 0.7)',
                borderColor: 'rgba(54, 162, 235, 1)',
                borderWidth: 1
            },
            {
                label: 'Absent',
                data: <?php echo json_encode($weekly_teacher_attendance_absent); ?>,
                backgroundColor: 'rgba(255, 159, 64, 0.7)',
                borderColor: 'rgba(255, 159, 64, 1)',
                borderWidth: 1
            }
        ]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
            y: {
                beginAtZero: true,
                title: {
                    display: true,
                    text: 'Number of Teachers'
                },
                ticks: {
                    stepSize: 1
                }
            },
            x: {
                title: {
                    display: true,
                    text: 'Date'
                }
            }
        }
    }
});
</script>

<?php include 'include/footer.php'; ?>