<?php
// paystack_redirect.php
ob_start();
session_start();

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header("Location: login.php");
    exit();
}

include '../config/conn.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate inputs
    $invoice_id = intval($_POST['invoice_id'] ?? 0);
    $email = filter_var($_POST['email'] ?? '', FILTER_SANITIZE_EMAIL);
    $amount = intval($_POST['amount'] ?? 0);
    
    error_log("=== PAYSTACK REDIRECT STARTED ===");
    error_log("Invoice ID: $invoice_id, Amount: $amount, Email: $email");
    
    // Basic validation
    if ($invoice_id <= 0) {
        $error = "Invalid invoice ID";
        error_log("ERROR: $error");
        header("Location: payment_failed.php?error=validation&message=" . urlencode($error));
        exit();
    }
    
    if ($amount <= 0) {
        $error = "Invalid amount";
        error_log("ERROR: $error");
        header("Location: payment_failed.php?error=validation&message=" . urlencode($error));
        exit();
    }
    
    // Ensure minimum amount (₦1 = 100 kobo)
    if ($amount < 100) {
        $amount = 100; // Set to minimum
        error_log("Amount adjusted to minimum: 100 kobo");
    }
    
    // Get invoice details
    $invoice_query = "SELECT i.*, s.full_name, s.email as student_email 
                      FROM invoices i 
                      JOIN students s ON i.student_id = s.id 
                      WHERE i.id = ?";
    $stmt = $conn->prepare($invoice_query);
    
    if (!$stmt) {
        $error = "Database prepare failed: " . $conn->error;
        error_log("ERROR: $error");
        header("Location: payment_failed.php?error=database&message=" . urlencode($error));
        exit();
    }
    
    $stmt->bind_param("i", $invoice_id);
    
    if (!$stmt->execute()) {
        $error = "Database execute failed: " . $stmt->error;
        error_log("ERROR: $error");
        header("Location: payment_failed.php?error=database&message=" . urlencode($error));
        exit();
    }
    
    $invoice_result = $stmt->get_result();
    
    if ($invoice_result->num_rows === 0) {
        $error = "Invoice not found: $invoice_id";
        error_log("ERROR: $error");
        header("Location: payment_failed.php?error=not_found&message=" . urlencode($error));
        exit();
    }
    
    $invoice = $invoice_result->fetch_assoc();
    $stmt->close();
    
    // Use student email if available, otherwise use provided email
    $payer_email = !empty($invoice['student_email']) ? $invoice['student_email'] : $email;
    
    // Validate email
    if (!filter_var($payer_email, FILTER_VALIDATE_EMAIL)) {
        // Use a default test email for Paystack
        $payer_email = 'customer@email.com';
        error_log("Email invalid, using default: $payer_email");
    }
    
    // Paystack configuration
    $paystack_secret = "sk_test_e961ecf6ab503d4b5d63853a2dfc0c9ec6ac78e6";
    $callback_url = "https://adinoheightschool.com/admin/payment_callback.php";
    
    // Generate unique reference
    $reference = 'INV_' . $invoice_id . '_' . uniqid();
    
    // Prepare request data
    $fields = [
        'email' => $payer_email,
        'amount' => $amount,
        'reference' => $reference,
        'callback_url' => $callback_url,
        'metadata' => [
            'invoice_id' => $invoice_id,
            'student_name' => $invoice['full_name']
        ]
    ];
    
    error_log("Paystack Request Data: " . json_encode($fields));
    
    // Initialize cURL
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => "https://api.paystack.co/transaction/initialize",
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($fields),
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $paystack_secret",
            "Content-Type: application/json",
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FAILONERROR => true
    ]);
    
    // Execute request
    $response = curl_exec($ch);
    $err = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    error_log("Paystack Response - HTTP Code: $http_code");
    error_log("cURL Error: " . $err);
    error_log("Response: " . $response);
    
    if ($err) {
        curl_close($ch);
        error_log("cURL Error occurred");
        header("Location: payment_failed.php?reference=$reference&error=connection&message=" . urlencode($err));
        exit();
    }
    
    curl_close($ch);
    
    // Parse response
    $res = json_decode($response, true);
    
    if (!$res) {
        error_log("Invalid JSON response");
        header("Location: payment_failed.php?reference=$reference&error=invalid_response&message=Invalid response from payment gateway");
        exit();
    }
    
    if (!$res['status']) {
        $error_message = $res['message'] ?? 'Unknown API error';
        error_log("Paystack API Error: $error_message");
        header("Location: payment_failed.php?reference=$reference&error=api&message=" . urlencode($error_message));
        exit();
    }
    
    // Success - store reference and redirect to Paystack
    $ref = $res['data']['reference'];
    $update_query = "UPDATE invoices SET reference=?, payment_method='paystack' WHERE id=?";
    $stmt = $conn->prepare($update_query);
    
    if ($stmt) {
        $stmt->bind_param("si", $ref, $invoice_id);
        $stmt->execute();
        $stmt->close();
    }
    
    error_log("Success! Redirecting to: " . $res['data']['authorization_url']);
    header("Location: " . $res['data']['authorization_url']);
    exit();
    
} else {
    // If accessed via GET, redirect back
    header("Location: view_invoices.php");
    exit();
}

ob_end_flush();
?>