<?php
// Enable output buffering
ob_start();
    include 'include/header.php'; 
// Secure session cookie parameters
session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
    'httponly' => true,
    'samesite' => 'Lax'
]);

// Start session


// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

// Check session expiry (1 hour)
$inactive = 3600;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > $inactive)) {
    session_unset();
    session_destroy();
    header("Location: ../login.php?timeout=1");
    exit();
}
$_SESSION['last_activity'] = time();

// Restrict to admin only
if (!isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'admin') {
    header("Location: ../login.php?access=denied");
    exit();
}

// ✅ Fetch admin details from database
include '../config/conn.php';

// Initialize variables
$success = '';
$error = '';

// Handle Excel Export
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['export_excel'])) {
    $quiz_id = intval($_POST['quiz_id']);
    
    // Get test details
    $test_query = "SELECT q.*, s.subject_name 
                  FROM quizzes q 
                  JOIN subjects s ON q.subject_id = s.id 
                  WHERE q.id = ?";
    $stmt = mysqli_prepare($conn, $test_query);
    mysqli_stmt_bind_param($stmt, "i", $quiz_id);
    mysqli_stmt_execute($stmt);
    $test_result = mysqli_stmt_get_result($stmt);
    $selected_test = mysqli_fetch_assoc($test_result);
    
    if (!$selected_test) {
        $error = "Test not found";
    } else {
        // Get all results for this test
        $results_query = "SELECT qr.*, s.full_name, s.student_id, c.class_name, sec.section_name
                         FROM quiz_results qr
                         JOIN students s ON qr.student_id = s.id
                         LEFT JOIN classes c ON s.class = c.id
                         LEFT JOIN sections sec ON s.section = sec.id
                         WHERE qr.quiz_id = ?
                         ORDER BY qr.percentage DESC";
        $stmt = mysqli_prepare($conn, $results_query);
        mysqli_stmt_bind_param($stmt, "i", $quiz_id);
        mysqli_stmt_execute($stmt);
        $results_result = mysqli_stmt_get_result($stmt);
        
        $test_results = [];
        if ($results_result) {
            while ($result = mysqli_fetch_assoc($results_result)) {
                $test_results[] = $result;
            }
        }
        
        // Generate Excel file
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment; filename="test_results_'.date('Y-m-d').'.xls"');
        
        echo '<table border="1">';
        echo '<tr><th colspan="8" style="background-color: #4F81BD; color: white; font-size: 16px; height: 30px;">Test Results: ' . htmlspecialchars($selected_test['quiz_title']) . '</th></tr>';
        echo '<tr><th colspan="8" style="background-color: #DCE6F1;">Subject: ' . htmlspecialchars($selected_test['subject_name']) . ' | Date: ' . date('M j, Y', strtotime($selected_test['start_date'])) . '</th></tr>';
        echo '<tr style="background-color: #B8CCE4; font-weight: bold;">';
        echo '<th>Student ID</th>';
        echo '<th>Student Name</th>';
        echo '<th>Class</th>';
        echo '<th>Score</th>';
        echo '<th>Percentage</th>';
        echo '<th>Time Taken (seconds)</th>';
        echo '<th>Status</th>';
        echo '<th>Submitted At</th>';
        echo '</tr>';
        
        foreach ($test_results as $result) {
            echo '<tr>';
            echo '<td>' . htmlspecialchars($result['student_id']) . '</td>';
            echo '<td>' . htmlspecialchars($result['full_name']) . '</td>';
            echo '<td>' . htmlspecialchars($result['class_name'] . ' ' . $result['section_name']) . '</td>';
            echo '<td>' . $result['score'] . '/' . $result['total_marks'] . '</td>';
            echo '<td>' . $result['percentage'] . '%</td>';
            echo '<td>' . $result['time_taken'] . '</td>';
            echo '<td>' . ($result['result_released'] ? 'Released' : 'Pending') . '</td>';
            echo '<td>' . date('M j, Y g:i A', strtotime($result['submitted_at'])) . '</td>';
            echo '</tr>';
        }
        
        echo '</table>';
        exit();
    }
}

// Handle score release
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['release_scores'])) {
    $quiz_id = intval($_POST['quiz_id']);
    
    // Update all results for this quiz to released status
    $update_query = "UPDATE quiz_results SET result_released = 1 WHERE quiz_id = ?";
    $stmt = mysqli_prepare($conn, $update_query);
    mysqli_stmt_bind_param($stmt, "i", $quiz_id);
    
    if (mysqli_stmt_execute($stmt)) {
        $success = "Scores for this test have been released to students.";
    } else {
        $error = "Error releasing scores: " . mysqli_error($conn);
    }
}

// Handle individual score release
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['release_individual'])) {
    $result_id = intval($_POST['result_id']);
    
    // Update this specific result to released status
    $update_query = "UPDATE quiz_results SET result_released = 1 WHERE id = ?";
    $stmt = mysqli_prepare($conn, $update_query);
    mysqli_stmt_bind_param($stmt, "i", $result_id);
    
    if (mysqli_stmt_execute($stmt)) {
        $success = "This student's score has been released.";
    } else {
        $error = "Error releasing score: " . mysqli_error($conn);
    }
}

// Handle individual score deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_individual'])) {
    $result_id = intval($_POST['result_id']);
    
    // Delete this specific result
    $delete_query = "DELETE FROM quiz_results WHERE id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $result_id);
    
    if (mysqli_stmt_execute($stmt)) {
        $success = "This student's score has been deleted.";
    } else {
        $error = "Error deleting score: " . mysqli_error($conn);
    }
}

// Handle all scores deletion for a test
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_all_scores'])) {
    $quiz_id = intval($_POST['quiz_id']);
    
    // Delete all results for this quiz
    $delete_query = "DELETE FROM quiz_results WHERE quiz_id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $quiz_id);
    
    if (mysqli_stmt_execute($stmt)) {
        $success = "All scores for this test have been deleted.";
    } else {
        $error = "Error deleting scores: " . mysqli_error($conn);
    }
}

// Fetch all tests
$tests_query = "SELECT q.*, s.subject_name, 
               (SELECT COUNT(*) FROM quiz_results WHERE quiz_id = q.id) as result_count,
               (SELECT COUNT(*) FROM quiz_results WHERE quiz_id = q.id AND result_released = 1) as released_count
               FROM quizzes q 
               JOIN subjects s ON q.subject_id = s.id 
               ORDER BY q.start_date DESC";
$tests_result = mysqli_query($conn, $tests_query);
$tests = [];

if ($tests_result) {
    while ($test = mysqli_fetch_assoc($tests_result)) {
        $tests[] = $test;
    }
} else {
    $error = "Error fetching tests: " . mysqli_error($conn);
}

// Get specific test results if requested
$test_results = [];
$selected_test = null;
if (isset($_GET['view_results']) && is_numeric($_GET['view_results'])) {
    $test_id = intval($_GET['view_results']);
    
    // Get test details
    $test_query = "SELECT q.*, s.subject_name 
                  FROM quizzes q 
                  JOIN subjects s ON q.subject_id = s.id 
                  WHERE q.id = ?";
    $stmt = mysqli_prepare($conn, $test_query);
    mysqli_stmt_bind_param($stmt, "i", $test_id);
    mysqli_stmt_execute($stmt);
    $test_result = mysqli_stmt_get_result($stmt);
    $selected_test = mysqli_fetch_assoc($test_result);
    
    if ($selected_test) {
        // Get all results for this test
        $results_query = "SELECT qr.*, s.full_name, s.student_id, c.class_name, sec.section_name
                         FROM quiz_results qr
                         JOIN students s ON qr.student_id = s.id
                         LEFT JOIN classes c ON s.class = c.id
                         LEFT JOIN sections sec ON s.section = sec.id
                         WHERE qr.quiz_id = ?
                         ORDER BY qr.percentage DESC";
        $stmt = mysqli_prepare($conn, $results_query);
        mysqli_stmt_bind_param($stmt, "i", $test_id);
        mysqli_stmt_execute($stmt);
        $results_result = mysqli_stmt_get_result($stmt);
        
        if ($results_result) {
            while ($result = mysqli_fetch_assoc($results_result)) {
                $test_results[] = $result;
            }
        } else {
            $error = "Error fetching results: " . mysqli_error($conn);
        }
    } else {
        $error = "Test not found";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Release Test Scores - Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        function confirmDelete(message) {
            return confirm(message);
        }
        
        function exportToExcel(quizId) {
            // Create a form and submit it to trigger the Excel export
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '';
            
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'export_excel';
            input.value = '1';
            form.appendChild(input);
            
            const quizInput = document.createElement('input');
            quizInput.type = 'hidden';
            quizInput.name = 'quiz_id';
            quizInput.value = quizId;
            form.appendChild(quizInput);
            
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }
    </script>
    <style>
        .score-cell {
            font-weight: 600;
        }
        .score-high {
            color: #10B981;
        }
        .score-medium {
            color: #F59E0B;
        }
        .score-low {
            color: #EF4444;
        }
        .table-row:hover {
            background-color: #F9FAFB;
        }
        .action-btn {
            transition: all 0.2s ease;
        }
        .action-btn:hover {
            transform: translateY(-1px);
        }
    </style>
</head>
<body class="bg-gray-50">

    
    <div class="container mx-auto px-4 py-6">
        <h1 class="text-2xl md:text-3xl font-bold text-gray-800 mb-2">Test Score Management</h1>
        <p class="text-gray-600 mb-6">Manage and release test scores to students</p>
        
        <?php if (!empty($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3  mb-6 flex items-center">
                <i class="fas fa-check-circle mr-2"></i> 
                <span><?php echo htmlspecialchars($success); ?></span>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3  mb-6 flex items-center">
                <i class="fas fa-exclamation-circle mr-2"></i> 
                <span><?php echo htmlspecialchars($error); ?></span>
            </div>
        <?php endif; ?>

        <!-- Test Selection -->
        <div class="bg-white  p-6 mb-6">
            <h2 class="text-xl font-bold text-gray-800 mb-4">Select Test to Manage Scores</h2>
            
            <?php if (empty($tests)): ?>
                <div class="text-center py-8">
                    <i class="fas fa-clipboard-list text-4xl text-gray-400 mb-4"></i>
                    <p class="text-gray-500">No tests available.</p>
                </div>
            <?php else: ?>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    <?php foreach ($tests as $test): ?>
                        <div class="border border-gray-200 -lg p-4 hover:shadow-lg transition-shadow duration-300">
                            <div class="flex justify-between items-start mb-3">
                                <h3 class="font-semibold text-lg text-gray-800"><?php echo htmlspecialchars($test['quiz_title']); ?></h3>
                                <span class="bg-blue-100 text-blue-800 text-xs font-medium px-2 py-1 ">
                                    <?php echo $test['released_count'] . '/' . $test['result_count']; ?> released
                                </span>
                            </div>
                            
                            <p class="text-gray-600 mb-2"><?php echo htmlspecialchars($test['subject_name']); ?></p>
                            <p class="text-sm text-gray-500 mb-2">Date: <?php echo date('M j, Y', strtotime($test['start_date'])); ?></p>
                            
                            <div class="flex flex-wrap justify-between mt-4 gap-2">
                                <a href="?view_results=<?php echo $test['id']; ?>" class="text-blue-600 hover:text-blue-800 text-sm flex items-center">
                                    <i class="fas fa-eye mr-1"></i> View Results
                                </a>
                                
                                <?php if ($test['result_count'] > 0): ?>
                                    <div class="flex space-x-2">
                                        <?php if ($test['released_count'] < $test['result_count']): ?>
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="quiz_id" value="<?php echo $test['id']; ?>">
                                                <button type="submit" name="release_scores" class="text-green-600 hover:text-green-800 text-sm"
                                                        onclick="return confirm('Release ALL scores for this test to students?')">
                                                    <i class="fas fa-paper-plane mr-1"></i> Release All
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        
                                       <a href="exportTest.php?quiz_id=<?php echo $test['id']; ?>" 
   class="text-purple-600 hover:text-purple-800 text-sm">
   <i class="fas fa-file-excel mr-1"></i> Export
</a>
                                        
                                        <form method="POST" class="inline">
                                            <input type="hidden" name="quiz_id" value="<?php echo $test['id']; ?>">
                                            <button type="submit" name="delete_all_scores" class="text-red-600 hover:text-red-800 text-sm"
                                                    onclick="return confirmDelete('WARNING: This will delete ALL scores for this test. This action cannot be undone. Continue?')">
                                                <i class="fas fa-trash-alt mr-1"></i> Delete All
                                            </button>
                                        </form>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- Test Results -->
        <?php if ($selected_test): ?>
            <?php if (!empty($test_results)): ?>
                <div class="bg-white -md p-6">
                    <div class="flex justify-between items-center mb-6">
                        <div>
                            <h2 class="text-xl font-bold text-gray-800">Test Results: <?php echo htmlspecialchars($selected_test['quiz_title']); ?></h2>
                            <p class="text-gray-600"><?php echo htmlspecialchars($selected_test['subject_name']); ?> | <?php echo date('M j, Y', strtotime($selected_test['start_date'])); ?></p>
                        </div>
                        <span class="bg-blue-100 text-blue-800 text-sm font-medium px-3 py-1 ">
                            <?php 
                            $released_count = 0;
                            foreach ($test_results as $result) {
                                if ($result['result_released']) $released_count++;
                            }
                            echo $released_count . '/' . count($test_results) . ' results released';
                            ?>
                        </span>
                    </div>
                    
                    <div class="mb-4 flex justify-between items-center">
                        <p class="text-gray-600">Total Students: <span class="font-semibold"><?php echo count($test_results); ?></span></p>
                     <a href="exportTest.php?quiz_id=<?php echo $selected_test['id']; ?>" 
   class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg flex items-center">
   <i class="fas fa-file-excel mr-2"></i> Export to Excel
</a>

                    </div>
                    
                    <div class="overflow-x-auto ">
                        <table class="min-w-full bg-white">
                            <thead class="bg-gray-200 text-gray-700">
                                <tr>
                                    <th class="py-3 px-4 text-left">Student</th>
                                    <th class="py-3 px-4 text-left">ID</th>
                                    <th class="py-3 px-4 text-left">Class</th>
                                    <th class="py-3 px-4 text-left">Score</th>
                                    <th class="py-3 px-4 text-left">Percentage</th>
                                    <th class="py-3 px-4 text-left">Time Taken</th>
                                    <th class="py-3 px-4 text-left">Status</th>
                                    <th class="py-3 px-4 text-left">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-600">
                                <?php foreach ($test_results as $result): 
                                    $score_class = 'score-high';
                                    if ($result['percentage'] < 70) $score_class = 'score-medium';
                                    if ($result['percentage'] < 50) $score_class = 'score-low';
                                ?>
                                    <tr class="border-b border-gray-200 table-row">
                                        <td class="py-3 px-4"><?php echo htmlspecialchars($result['full_name']); ?></td>
                                        <td class="py-3 px-4"><?php echo htmlspecialchars($result['student_id']); ?></td>
                                        <td class="py-3 px-4"><?php echo htmlspecialchars($result['class_name'] . ' ' . $result['section_name']); ?></td>
                                        <td class="py-3 px-4 score-cell <?php echo $score_class; ?>"><?php echo $result['score'] . '/' . $result['total_marks']; ?></td>
                                        <td class="py-3 px-4 score-cell <?php echo $score_class; ?>"><?php echo $result['percentage']; ?>%</td>
                                        <td class="py-3 px-4"><?php echo gmdate("i:s", $result['time_taken']); ?></td>
                                        <td class="py-3 px-4">
                                            <span class="px-2 py-1 text-xs font-medium -full 
                                                <?php echo $result['result_released'] ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                                <?php echo $result['result_released'] ? 'Released' : 'Pending'; ?>
                                            </span>
                                        </td>
                                        <td class="py-3 px-4">
                                            <div class="flex space-x-2">
                                                <?php if (!$result['result_released']): ?>
                                                    <form method="POST" class="inline">
                                                        <input type="hidden" name="result_id" value="<?php echo $result['id']; ?>">
                                                        <button type="submit" name="release_individual" class="text-green-600 hover:text-green-800 text-sm action-btn"
                                                                onclick="return confirm('Release this score to the student?')">
                                                            <i class="fas fa-paper-plane mr-1"></i> Release
                                                        </button>
                                                    </form>
                                                <?php else: ?>
                                                    <span class="text-green-600 text-sm">
                                                        <i class="fas fa-check mr-1"></i> Released
                                                    </span>
                                                <?php endif; ?>
                                                
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="result_id" value="<?php echo $result['id']; ?>">
                                                    <button type="submit" name="delete_individual" class="text-red-600 hover:text-red-800 text-sm action-btn"
                                                            onclick="return confirmDelete('Are you sure you want to delete this student\'s score? This action cannot be undone.')">
                                                        <i class="fas fa-trash-alt mr-1"></i> Delete
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="mt-6 flex justify-between">
                        <a href="release.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 -lg action-btn flex items-center">
                            <i class="fas fa-arrow-left mr-2"></i> Back to Tests
                        </a>
                        
                        <div class="flex space-x-2">
                            <?php if ($released_count < count($test_results)): ?>
                                <form method="POST" class="inline">
                                    <input type="hidden" name="quiz_id" value="<?php echo $selected_test['id']; ?>">
                                    <button type="submit" name="release_scores" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 -lg action-btn flex items-center"
                                            onclick="return confirm('Release ALL scores for this test to students?')">
                                        <i class="fas fa-paper-plane mr-2"></i> Release All Remaining
                                    </button>
                                </form>
                            <?php endif; ?>
                            
                            <?php if (count($test_results) > 0): ?>
                                <form method="POST" class="inline">
                                    <input type="hidden" name="quiz_id" value="<?php echo $selected_test['id']; ?>">
                                    <button type="submit" name="delete_all_scores" class="bg-red-600 hover:bg-red-700 text-white px-4 py-2 -lg action-btn flex items-center"
                                            onclick="return confirmDelete('WARNING: This will delete ALL scores for this test. This action cannot be undone. Continue?')">
                                        <i class="fas fa-trash-alt mr-2"></i> Delete All Scores
                                    </button>
                                </form>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <div class="bg-white -md p-6 text-center">
                    <i class="fas fa-clipboard-list text-4xl text-gray-400 mb-4"></i>
                    <h3 class="text-xl font-semibold text-gray-700 mb-2">No Results Available</h3>
                    <p class="text-gray-500">No students have taken this test yet.</p>
                    <a href="release.php" class="inline-block mt-4 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 -lg action-btn">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Tests
                    </a>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    
    <?php include 'include/footer.php'; ?>
</body>
</html>