<?php
// Include header
include 'include/header.php';

// Check if user is logged in, if not redirect to login page
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type'])) {
    header('Location: ../portal_login.php');
    exit();
}

// Check if user is a student, if not redirect to appropriate dashboard
if ($_SESSION['user_type'] !== 'student') {
    if ($_SESSION['user_type'] === 'teacher') {
        header('Location: teacher_dashboard.php');
    } else if ($_SESSION['user_type'] === 'parent') {
        header('Location: parent_dashboard.php');
    }
    exit();
}

// Include database connection
include '../config/conn.php';

// Get student data including class and section
$student_id = $_SESSION['user_id'];
$query = "SELECT s.*, c.class_name, sec.section_name, c.id as class_id, sec.id as section_id
          FROM students s 
          LEFT JOIN classes c ON s.class = c.id 
          LEFT JOIN sections sec ON s.section = sec.id 
          WHERE s.id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("s", $student_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // Student not found in database, logout user
    session_destroy();
    header('Location: ../portal_login.php');
    exit();
}

$student = $result->fetch_assoc();
$stmt->close();

// Get the student's class and section IDs
$class_id = $student['class_id'];
$section_id = $student['section_id'];

// Initialize variables
$month = isset($_GET['month']) ? $_GET['month'] : date('Y-m');
$attendance_date = isset($_GET['attendance_date']) ? $_GET['attendance_date'] : date('Y-m-d');

// Get current month and year for calendar
$first_day = date('N', strtotime($month . '-01'));
$days_in_month = date('t', strtotime($month . '-01'));
$month_name = date('F Y', strtotime($month . '-01'));

// Get attendance data for the student for the selected month
$attendance_data = [];
$start_date = date('Y-m-01', strtotime($month));
$end_date = date('Y-m-t', strtotime($month));

$attendance_query = "SELECT attendance_date, status 
                     FROM attendance 
                     WHERE student_id = ? 
                     AND attendance_date BETWEEN ? AND ?
                     ORDER BY attendance_date";
$attendance_stmt = $conn->prepare($attendance_query);
$attendance_stmt->bind_param("iss", $student_id, $start_date, $end_date);
$attendance_stmt->execute();
$attendance_result = $attendance_stmt->get_result();

while ($row = $attendance_result->fetch_assoc()) {
    $attendance_data[$row['attendance_date']] = $row['status'];
}
$attendance_stmt->close();

// Calculate attendance statistics
$total_days = 0;
$present_days = 0;
$absent_days = 0;
$late_days = 0;
$half_days = 0;

foreach ($attendance_data as $status) {
    $total_days++;
    switch ($status) {
        case 'present':
            $present_days++;
            break;
        case 'absent':
            $absent_days++;
            break;
        case 'late':
            $late_days++;
            break;
        case 'half_day':
            $half_days++;
            break;
    }
}

$attendance_percentage = $total_days > 0 ? round(($present_days + ($late_days * 0.5) + ($half_days * 0.5)) / $total_days * 100, 2) : 0;

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: ../portal_login.php');
    exit();
}


?>
<style>
 @media (max-width: 600px) {
        #forms { /* match the id, not class */
            display: grid;
            grid-template-columns: 1fr; /* stack items */
            gap: 10px; /* add spacing between inputs */
        }
    }
</style>
<div class="bg-white p-0  ">
   

    <!-- Student Information -->
    <div class="bg-blue-50 p-0  mb-6">
        <h3 class="text-lg font-semibold text-blue-800">
            <i class="fas fa-user-graduate mr-2"></i>
            <?php echo htmlspecialchars($student['full_name']); ?>
        </h3>
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mt-2">
            <div class="text-blue-600">
                <i class="fas fa-id-card mr-1"></i> 
                ID: <?php echo htmlspecialchars($student['student_id']); ?>
            </div>
            <div class="text-blue-600">
                <i class="fas fa-graduation-cap mr-1"></i> 
                Class: <?php echo htmlspecialchars($student['class_name']); ?>
            </div>
            <div class="text-blue-600">
                <i class="fas fa-layer-group mr-1"></i> 
                Section: <?php echo htmlspecialchars($student['section_name']); ?>
            </div>
            <div class="text-blue-600">
                <i class="fas fa-hashtag mr-1"></i> 
                Roll: <?php echo htmlspecialchars($student['roll_number']); ?>
            </div>
        </div>
    </div>

    <!-- Attendance Statistics -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 mb-6">
        <div class="bg-green-50 p-4  border border-green-200">
            <div class="text-2xl font-bold text-green-800"><?php echo $present_days; ?></div>
            <div class="text-sm text-green-600">Present Days</div>
        </div>
        <div class="bg-red-50 p-4  border border-red-200">
            <div class="text-2xl font-bold text-red-800"><?php echo $absent_days; ?></div>
            <div class="text-sm text-red-600">Absent Days</div>
        </div>
        <div class="bg-yellow-50 p-4  border border-yellow-200">
            <div class="text-2xl font-bold text-yellow-800"><?php echo $late_days; ?></div>
            <div class="text-sm text-yellow-600">Late Days</div>
        </div>
        <div class="bg-blue-50 p-4  border border-blue-200">
            <div class="text-2xl font-bold text-blue-800"><?php echo $half_days; ?></div>
            <div class="text-sm text-blue-600">Half Days</div>
        </div>
        <div class="bg-purple-50 p-4  border border-purple-200">
            <div class="text-2xl font-bold text-purple-800"><?php echo $attendance_percentage; ?>%</div>
            <div class="text-sm text-purple-600">Attendance %</div>
        </div>
    </div>

    <!-- Month Selection -->
    <div class="bg-gray-50 p-4  mb-6">
        <h3 class="text-lg font-medium text-gray-800 mb-3">
            <i class="fas fa-calendar mr-2"></i> View Attendance for Month
        </h3>
        <form method="GET" action="" class="flex items-center space-x-4" id="forms">
            <input type="month" name="month" value="<?php echo $month; ?>" 
                   class="px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500">
            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 ">
                <i class="fas fa-eye mr-2"></i> View
            </button>
            <a href="attendance.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 ">
                <i class="fas fa-sync mr-2"></i> Current Month
            </a>
        </form>
    </div>

    <!-- Attendance Calendar -->
    <div class="bg-white p-0   mb-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">
            <i class="fas fa-calendar-check mr-2"></i> 
            Attendance for <?php echo $month_name; ?>
        </h3>
        
        <div class="grid grid-cols-7 gap-2 mb-2">
            <div class="text-center font-medium text-gray-700">Mon</div>
            <div class="text-center font-medium text-gray-700">Tue</div>
            <div class="text-center font-medium text-gray-700">Wed</div>
            <div class="text-center font-medium text-gray-700">Thu</div>
            <div class="text-center font-medium text-gray-700">Fri</div>
            <div class="text-center font-medium text-gray-700">Sat</div>
            <div class="text-center font-medium text-gray-700">Sun</div>
        </div>
        
        <div class="grid grid-cols-7 gap-2">
            <?php
            // Add empty cells for days before the first day of the month
            for ($i = 1; $i < $first_day; $i++) {
                echo '<div class="h-16 border  bg-gray-100"></div>';
            }
            
            // Add cells for each day of the month
            for ($day = 1; $day <= $days_in_month; $day++) {
                $current_date = date('Y-m-d', strtotime($month . '-' . sprintf('%02d', $day)));
                $is_today = ($current_date == date('Y-m-d')) ? 'border-2 border-blue-500' : 'border';
                $has_attendance = isset($attendance_data[$current_date]);
                $day_of_week = date('N', strtotime($current_date));
                
                // Check if it's a weekend (Saturday = 6, Sunday = 7)
                $is_weekend = ($day_of_week >= 6);
                
                if ($has_attendance) {
                    $status = $attendance_data[$current_date];
                    switch ($status) {
                        case 'present':
                            $bg_color = 'bg-green-100';
                            $status_icon = 'fa-check-circle';
                            $status_text = 'Present';
                            break;
                        case 'absent':
                            $bg_color = 'bg-red-100';
                            $status_icon = 'fa-times-circle';
                            $status_text = 'Absent';
                            break;
                        case 'late':
                            $bg_color = 'bg-yellow-100';
                            $status_icon = 'fa-clock';
                            $status_text = 'Late';
                            break;
                        case 'half_day':
                            $bg_color = 'bg-blue-100';
                            $status_icon = 'fa-adjust';
                            $status_text = 'Half Day';
                            break;
                        default:
                            $bg_color = 'bg-gray-50';
                            $status_icon = 'fa-question-circle';
                            $status_text = 'Unknown';
                    }
                } else if ($is_weekend) {
                    $bg_color = 'bg-gray-100';
                    $status_icon = 'fa-home';
                    $status_text = 'Weekend';
                } else {
                    $bg_color = 'bg-gray-50';
                    $status_icon = 'fa-minus-circle';
                    $status_text = 'No Record';
                }
                
                echo '<div class="h-16  p-1 ' . $bg_color . ' ' . $is_today . '" title="' . date('D, M j, Y', strtotime($current_date)) . ' - ' . $status_text . '">';
                echo '<div class="text-sm font-medium">' . $day . '</div>';
                echo '<div class="text-xs mt-1"><i class="fas ' . $status_icon . '"></i></div>';
                
                if ($has_attendance) {
                    echo '<div class="text-xs">' . ucfirst($status) . '</div>';
                }
                
                echo '</div>';
                
                // Start new row after Sunday
                if (($first_day + $day - 1) % 7 == 0 && $day != $days_in_month) {
                    echo '</div><div class="grid grid-cols-7 gap-2">';
                }
            }
            
            // Add empty cells for remaining days in the last week
            $last_day = date('N', strtotime($month . '-' . $days_in_month));
            if ($last_day < 7) {
                for ($i = $last_day; $i < 7; $i++) {
                    echo '<div class="h-16 border  bg-gray-100"></div>';
                }
            }
            ?>
        </div>
        
        <!-- Legend -->
        <div class="mt-4 grid grid-cols-2 md:grid-cols-5 gap-2 text-xs">
            <div class="flex items-center">
                <div class="w-4 h-4 bg-green-100 mr-1 border border-green-200"></div>
                <span>Present</span>
            </div>
            <div class="flex items-center">
                <div class="w-4 h-4 bg-red-100 mr-1 border border-red-200"></div>
                <span>Absent</span>
            </div>
            <div class="flex items-center">
                <div class="w-4 h-4 bg-yellow-100 mr-1 border border-yellow-200"></div>
                <span>Late</span>
            </div>
            <div class="flex items-center">
                <div class="w-4 h-4 bg-blue-100 mr-1 border border-blue-200"></div>
                <span>Half Day</span>
            </div>
            <div class="flex items-center">
                <div class="w-4 h-4 bg-gray-100 mr-1 border border-gray-200"></div>
                <span>Weekend/No Record</span>
            </div>
        </div>
    </div>

    <!-- Detailed Attendance Records -->
    <div class="bg-white p-0 ">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">
            <i class="fas fa-list-alt mr-2"></i> 
            Detailed Attendance Records
        </h3>
        
        <?php if (!empty($attendance_data)): ?>
            <div class="overflow-x-auto">
                <table class="min-w-full bg-white">
                    <thead class="bg-gray-200 text-gray-700">
                        <tr>
                            <th class="py-3 px-4 text-left">Date</th>
                            <th class="py-3 px-4 text-left">Day</th>
                            <th class="py-3 px-4 text-left">Status</th>
                            <th class="py-3 px-4 text-left">Remarks</th>
                        </tr>
                    </thead>
                    <tbody class="text-gray-600">
                        <?php 
                        // Sort attendance data by date descending
                        krsort($attendance_data);
                        foreach ($attendance_data as $date => $status): 
                            $day_name = date('l', strtotime($date));
                            $is_weekend = (date('N', strtotime($date)) >= 6);
                            
                            switch ($status) {
                                case 'present':
                                    $status_class = 'bg-green-100 text-green-800';
                                    break;
                                case 'absent':
                                    $status_class = 'bg-red-100 text-red-800';
                                    break;
                                case 'late':
                                    $status_class = 'bg-yellow-100 text-yellow-800';
                                    break;
                                case 'half_day':
                                    $status_class = 'bg-blue-100 text-blue-800';
                                    break;
                                default:
                                    $status_class = 'bg-gray-100 text-gray-800';
                            }
                        ?>
                            <tr class="border-b border-gray-200 hover:bg-gray-50">
                                <td class="py-3 px-4"><?php echo date('M j, Y', strtotime($date)); ?></td>
                                <td class="py-3 px-4"><?php echo $day_name; ?></td>
                                <td class="py-3 px-4">
                                    <span class="px-2 py-1 rounded-full text-xs font-medium <?php echo $status_class; ?>">
                                        <?php echo ucfirst($status); ?>
                                    </span>
                                </td>
                                <td class="py-3 px-4">
                                    <?php if ($is_weekend): ?>
                                        <span class="text-gray-500">Weekend</span>
                                    <?php else: ?>
                                        <span class="text-gray-500">School Day</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="text-center py- 0text-gray-500">
                <i class="fas fa-calendar-times text-4xl mb-3"></i>
                <p>No attendance records found for <?php echo $month_name; ?>.</p>
            </div>
        <?php endif; ?>
    </div>

    <!-- Attendance Summary -->
    <div class="mt-6 bg-gray-50 p-0 ">
        <h4 class="text-md font-medium text-gray-700 mb-3">
            <i class="fas fa-chart-pie mr-2"></i> Attendance Summary
        </h4>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div class="bg-white p-4  ">
                <h5 class="font-medium text-gray-700 mb-2">Monthly Summary</h5>
                <div class="space-y-2">
                    <div class="flex justify-between">
                        <span>Total School Days:</span>
                        <span class="font-medium"><?php echo $total_days; ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span>Days Present:</span>
                        <span class="font-medium text-green-600"><?php echo $present_days; ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span>Days Absent:</span>
                        <span class="font-medium text-red-600"><?php echo $absent_days; ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span>Days Late:</span>
                        <span class="font-medium text-yellow-600"><?php echo $late_days; ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span>Half Days:</span>
                        <span class="font-medium text-blue-600"><?php echo $half_days; ?></span>
                    </div>
                </div>
            </div>
            <div class="bg-white p-0  ">
                <h5 class="font-medium text-gray-700 mb-2">Attendance Percentage</h5>
                <div class="text-center">
                    <div class="text-3xl font-bold text-purple-600 mb-2"><?php echo $attendance_percentage; ?>%</div>
                    <div class="w-full bg-gray-200 rounded-full h-4">
                        <div class="bg-purple-600 h-4 rounded-full" 
                             style="width: <?php echo min($attendance_percentage, 100); ?>%"></div>
                    </div>
                    <p class="text-sm text-gray-500 mt-2">
                        Based on <?php echo $total_days; ?> school days in <?php echo date('F', strtotime($month)); ?>
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Update page title with current month
document.addEventListener('DOMContentLoaded', function() {
    document.title = "My Attendance - <?php echo $month_name; ?> | Student Portal";
    
    // Highlight today's date in the calendar
    const today = new Date().toLocaleDateString('en-US', { 
        year: 'numeric', 
        month: '2-digit', 
        day: '2-digit' 
    }).replace(/(\d+)\/(\d+)\/(\d+)/, '$3-$1-$2');
    
    const calendarDays = document.querySelectorAll('.grid.grid-cols-7.gap-2 > div');
    calendarDays.forEach(day => {
        const dayNumber = day.querySelector('.text-sm.font-medium');
        if (dayNumber) {
            const dayDate = '<?php echo $month; ?>-' + dayNumber.textContent.padStart(2, '0');
            if (dayDate === today) {
                day.classList.add('ring-2', 'ring-blue-500');
            }
        }
    });
});
</script>

<?php include 'include/footer.php'; ?>