<?php
date_default_timezone_set('Africa/Lagos');
// Set consistent session configuration
session_name('ADINOHEIGHT_SESSION');
ini_set('session.cookie_lifetime', 0);
ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on');
ini_set('session.cookie_httponly', true);
ini_set('session.cookie_samesite', 'Lax');

ob_start();
session_start();
include '../config/conn.php';

// Enhanced check if user is logged in
if (!isset($_SESSION['user_id']) || !isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // Use absolute path for redirection
    header('Location: /portal/login.php');
    exit();
}

$student_id = $_SESSION['user_id'];
$success = '';
$error = '';

// Fetch student details
$student_query = "SELECT s.*, c.class_name, sec.section_name 
                  FROM students s 
                  LEFT JOIN classes c ON s.class = c.id 
                  LEFT JOIN sections sec ON s.section = sec.id 
                  WHERE s.id = $student_id";
$student_result = mysqli_query($conn, $student_query);
$student = mysqli_fetch_assoc($student_result);

// Get current date and time
$current_datetime = date('Y-m-d H:i:s');
$current_timestamp = time();

// Handle test submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_test'])) {
    $quiz_id = intval($_POST['quiz_id']);
    $time_taken = intval($_POST['time_taken']);
    
    // Check if student has already taken this test
    $check_query = "SELECT id FROM quiz_results WHERE quiz_id = $quiz_id AND student_id = $student_id";
    $check_result = mysqli_query($conn, $check_query);
    
    if (mysqli_num_rows($check_result) > 0) {
        $error = "You have already taken this test.";
    } else {
        // Calculate score
        $score = 0;
        $total_marks = 0;
        $essay_questions = []; // Track essay questions for manual grading

        $questions_query = "SELECT id, marks, question_type FROM quiz_questions WHERE quiz_id = $quiz_id";
        $questions_result = mysqli_query($conn, $questions_query);

        if ($questions_result) {
            while ($question = mysqli_fetch_assoc($questions_result)) {
                $question_id = $question['id'];
                $total_marks += $question['marks'];
                
                if (!isset($_POST['answer_'.$question_id]) || empty(trim($_POST['answer_'.$question_id]))) {
                    continue; // Skip unanswered questions
                }
                
                $student_answer = trim($_POST['answer_'.$question_id]);
                
                if ($question['question_type'] == 'multiple_choice') {
                    $selected_option = intval($student_answer);
                    
                    // Check if selected option is correct
                    $check_option = "SELECT is_correct FROM quiz_question_options 
                                    WHERE id = $selected_option AND question_id = $question_id";
                    $option_result = mysqli_query($conn, $check_option);
                    
                    if ($option_result && mysqli_num_rows($option_result) > 0) {
                        $option = mysqli_fetch_assoc($option_result);
                        
                        $is_correct = $option['is_correct'];
                        $is_correct_value = false;
                        
                        if (is_numeric($is_correct)) {
                            $is_correct_value = (int)$is_correct === 1;
                        } elseif (is_string($is_correct)) {
                            $is_correct_value = ($is_correct === '1' || strtolower($is_correct) === 'true');
                        } else {
                            $is_correct_value = (bool)$is_correct;
                        }
                        
                        if ($is_correct_value) {
                            $score += $question['marks'];
                        }
                    }
                }
                elseif ($question['question_type'] == 'true_false') {
                    // Get correct answer from database
                    $correct_answer_query = "SELECT option_text FROM quiz_question_options 
                                            WHERE question_id = $question_id AND is_correct = 1 
                                            LIMIT 1";
                    $correct_result = mysqli_query($conn, $correct_answer_query);
                    
                    if ($correct_result && mysqli_num_rows($correct_result) > 0) {
                        $correct_row = mysqli_fetch_assoc($correct_result);
                        $correct_answer = strtolower(trim($correct_row['option_text']));
                        
                        // Compare answers (case-insensitive)
                        if (strtolower($student_answer) === $correct_answer) {
                            $score += $question['marks'];
                        }
                    }
                }
                elseif ($question['question_type'] == 'short_answer') {
                    // Get correct answers from database
                    $correct_answers_query = "SELECT option_text FROM quiz_question_options 
                                             WHERE question_id = $question_id AND is_correct = 1";
                    $correct_result = mysqli_query($conn, $correct_answers_query);
                    
                    $is_correct = false;
                    
                    if ($correct_result && mysqli_num_rows($correct_result) > 0) {
                        while ($correct_row = mysqli_fetch_assoc($correct_result)) {
                            $correct_answer = trim($correct_row['option_text']);
                            
                            // Use similarity comparison instead of exact match
                            similar_text(strtolower($student_answer), strtolower($correct_answer), $similarity);
                            
                            if ($similarity >= 80) { // 80% similarity threshold
                                $is_correct = true;
                                break;
                            }
                        }
                    }
                    
                    if ($is_correct) {
                        $score += $question['marks'];
                    }
                }
                elseif ($question['question_type'] == 'essay') {
                    // Essay questions require manual grading
                    $essay_questions[] = [
                        'question_id' => $question_id,
                        'answer' => $student_answer,
                        'marks' => $question['marks']
                    ];
                    // Don't add to score automatically
                }
            }
        }

        // Calculate percentage based on auto-graded questions only
        $auto_graded_max = $total_marks;
        foreach ($essay_questions as $essay) {
            $auto_graded_max -= $essay['marks'];
        }

        $percentage = $auto_graded_max > 0 ? round(($score / $auto_graded_max) * 100, 2) : 0;
        
        // Insert into quiz_results
        $insert_query = "INSERT INTO quiz_results 
                        (quiz_id, student_id, score, total_marks, percentage, time_taken, submitted_at) 
                         VALUES ($quiz_id, $student_id, $score, $total_marks, $percentage, $time_taken, NOW())";
        
        if (mysqli_query($conn, $insert_query)) {
            $result_id = mysqli_insert_id($conn);
            
            // Save individual answers
            foreach ($_POST as $key => $value) {
                if (strpos($key, 'answer_') === 0) {
                    $q_id = intval(str_replace('answer_', '', $key));
                    
                    $question_type_query = "SELECT question_type FROM quiz_questions WHERE id = $q_id";
                    $type_result = mysqli_query($conn, $question_type_query);
                    
                    if ($type_result && mysqli_num_rows($type_result) > 0) {
                        $question = mysqli_fetch_assoc($type_result);
                        $question_type = $question['question_type'];
                        
                        if ($question_type == 'multiple_choice') {
                            $option_id = intval($value);
                            $answer_query = "INSERT INTO quiz_answers (result_id, question_id, option_id) 
                                             VALUES ($result_id, $q_id, $option_id)";
                        } else {
                            $answer_text = mysqli_real_escape_string($conn, $value);
                            $answer_query = "INSERT INTO quiz_answers (result_id, question_id, answer_text) 
                                             VALUES ($result_id, $q_id, '$answer_text')";
                        }
                        
                        mysqli_query($conn, $answer_query);
                    }
                }
            }
            
            $success = "Test submitted successfully! Your score will be available when released.";
        } else {
            $error = "Error submitting test: " . mysqli_error($conn);
        }
    }
}

// Fetch available tests for this student
$section_condition = "AND (q.section_id = 0";
if (isset($student['section']) && !empty($student['section'])) {
    $section_condition .= " OR q.section_id = " . intval($student['section']);
} else {
    $section_condition .= " OR q.section_id IS NULL";
}
$section_condition .= ")";

$tests_query = "SELECT q.*, s.subject_name, 
                (SELECT COUNT(*) FROM quiz_questions WHERE quiz_id = q.id) as question_count,
                (SELECT id FROM quiz_results WHERE quiz_id = q.id AND student_id = $student_id) as has_result,
                (SELECT result_released FROM quiz_results WHERE quiz_id = q.id AND student_id = $student_id) as result_released
                FROM quizzes q 
                JOIN subjects s ON q.subject_id = s.id 
                WHERE q.class_id = " . intval($student['class']) . " 
                $section_condition 
                AND q.status = 'active'
                ORDER BY q.start_date DESC";

$tests_result = mysqli_query($conn, $tests_query);
$available_tests = [];
$upcoming_tests = [];
$completed_tests = [];
$all_tests = [];

if ($tests_result) {
    while ($test = mysqli_fetch_assoc($tests_result)) {
        $test_status = 'upcoming';
        $can_take = false;
        $can_view = false;
        
        $current_timestamp = time();
        $start_timestamp = strtotime($test['start_date']);
        $end_timestamp = $test['end_date'] ? strtotime($test['end_date']) : null;
        
        // Store test in all_tests array for individual access
        $all_tests[$test['id']] = $test;
        
        // STRICT TIME CHECK: Test is only active if current time is AFTER start time
        if ($current_timestamp >= $start_timestamp && 
            ($end_timestamp === null || $current_timestamp <= $end_timestamp)) {
            $test_status = 'active';
            if (empty($test['has_result'])) {
                $can_take = true;
            }
        } elseif ($end_timestamp && $current_timestamp > $end_timestamp) {
            $test_status = 'ended';
            if (!empty($test['has_result']) && $test['result_released']) {
                $can_view = true;
            }
        } else {
            // Test is still upcoming (current time is BEFORE start time)
            $test_status = 'upcoming';
            $can_take = false;
        }
        
        if (!empty($test['has_result'])) {
            $test_status = 'completed';
            if ($test['result_released']) {
                $can_view = true;
            }
        }
        
        $test['status'] = $test_status;
        $test['can_take'] = $can_take;
        $test['can_view'] = $can_view;
        
        if ($test_status == 'active' && $can_take) {
            $available_tests[] = $test;
        } elseif ($test_status == 'upcoming') {
            $upcoming_tests[] = $test;
        } elseif ($test_status == 'completed' || $test_status == 'ended') {
            $completed_tests[] = $test;
        }
    }
} else {
    $error = "Error fetching tests: " . mysqli_error($conn);
}

// Get specific test if requested - FIXED LOGIC
$current_test = null;
$test_questions = [];
if (isset($_GET['take_test']) && is_numeric($_GET['take_test'])) {
    $test_id = intval($_GET['take_test']);
    
    // Check if test exists in all_tests array
    if (isset($all_tests[$test_id])) {
        $test = $all_tests[$test_id];
        
        // STRICT TIME CHECK: Prevent access if current time is BEFORE start time
        $current_timestamp = time();
        $start_timestamp = strtotime($test['start_date']);
        $end_timestamp = $test['end_date'] ? strtotime($test['end_date']) : null;
        
        // Check if test is currently active
        if ($current_timestamp >= $start_timestamp && 
            ($end_timestamp === null || $current_timestamp <= $end_timestamp)) {
            
            // Check if student has already taken this test
            if (empty($test['has_result'])) {
                $current_test = $test;
                
                // Fetch questions for the test
                $questions_query = "SELECT qq.* FROM quiz_questions qq 
                                   WHERE qq.quiz_id = $test_id 
                                   ORDER BY qq.id";
                $questions_result = mysqli_query($conn, $questions_query);
                
                if ($questions_result) {
                    while ($question = mysqli_fetch_assoc($questions_result)) {
                        if ($question['question_type'] == 'multiple_choice') {
                            $options_query = "SELECT * FROM quiz_question_options 
                                             WHERE question_id = {$question['id']} 
                                             ORDER BY id";
                            $options_result = mysqli_query($conn, $options_query);
                            $question['options'] = [];
                            
                            if ($options_result) {
                                while ($option = mysqli_fetch_assoc($options_result)) {
                                    $question['options'][] = $option;
                                }
                            }
                        }
                        $test_questions[] = $question;
                    }
                } else {
                    $error = "Error fetching questions: " . mysqli_error($conn);
                }
            } else {
                $error = "You have already taken this test.";
            }
        } elseif ($current_timestamp < $start_timestamp) {
            $time_until_start = $start_timestamp - $current_timestamp;
            $minutes_until_start = ceil($time_until_start / 60);
            
            $error = "This test is not available yet. It will start in " . $minutes_until_start . " minutes on " . 
                    date('M j, Y g:i A', $start_timestamp) . ". Please wait until the scheduled time.";
        } else {
            $error = "This test has ended and is no longer available.";
        }
    } else {
        $error = "Test not found or you don't have permission to access it.";
    }
}

// Get test results if requested
$test_result = null;
$test_details = null;
if (isset($_GET['view_result']) && is_numeric($_GET['view_result'])) {
    $test_id = intval($_GET['view_result']);
    
    foreach ($completed_tests as $test) {
        if ($test['id'] == $test_id && $test['can_view']) {
            $result_query = "SELECT * FROM quiz_results 
                            WHERE quiz_id = $test_id AND student_id = $student_id";
            $result_result = mysqli_query($conn, $result_query);
            if ($result_result) {
                $test_result = mysqli_fetch_assoc($result_result);
            }
            
            $test_details_query = "SELECT q.*, s.subject_name 
                                  FROM quizzes q 
                                  JOIN subjects s ON q.subject_id = s.id 
                                  WHERE q.id = $test_id";
            $test_details_result = mysqli_query($conn, $test_details_query);
            if ($test_details_result) {
                $test_details = mysqli_fetch_assoc($test_details_result);
            }
            break;
        }
    }
    
    if (!$test_result) {
        $error = "Results for this test are not available yet.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CBT Test Portal - Student</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .question-container {
            display: none;
        }
        .question-container.active {
            display: block;
        }
        .question-navigation {
            display: flex;
            flex-wrap: wrap;
            gap: 5px;
            margin-bottom: 20px;
        }
        .question-number {
            width: 35px;
            height: 35px;
            display: flex;
            align-items: center;
            justify-content: center;
            border: 2px solid #e5e7eb;
            border-radius: 5px;
            cursor: pointer;
            font-weight: 500;
        }
        .question-number.current {
            border-color: #3b82f6;
            background-color: #3b82f6;
            color: white;
        }
        .question-number.answered {
            border-color: #10b981;
            background-color: #10b981;
            color: white;
        }
        .question-number.marked {
            border-color: #f59e0b;
            background-color: #f59e0b;
            color: white;
        }
        .test-countdown {
            background: linear-gradient(45deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 10px;
            border-radius: 8px;
            text-align: center;
            margin: 10px 0;
        }
    </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-0 py-0">
        <!-- Header -->
        <div class="bg-white shadow p-2 mb-6">
            <div class="flex justify-between items-center">
                <div>
                    <h1 class="text-2xl font-bold text-gray-800">Computer-Based Test Portal</h1>
                    <p class="text-gray-600">Welcome, <?php echo htmlspecialchars($student['full_name'] . ' ' . $student['student_id']); ?></p>
                    <p class="text-sm text-gray-500"><?php echo htmlspecialchars($student['class_name'] . ' - ' . $student['section_name']); ?></p>
                </div>
                <div class="text-right">
                    <p class="text-gray-600" id="current-time"><?php echo date('l, F j, Y H:i:s'); ?></p>
                    <a href="logout.php" class="text-red-600 hover:text-red-800 text-sm">
                        <i class="fas fa-sign-out-alt mr-1"></i> Logout
                    </a>
                </div>
            </div>
        </div>

        <?php if (!empty($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-6">
                <i class="fas fa-check-circle mr-2"></i> <?php echo htmlspecialchars($success); ?>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
                <i class="fas fa-exclamation-circle mr-2"></i> <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <!-- Test Taking Interface -->
        <?php if ($current_test && !empty($test_questions)): ?>
            <div class="bg-white rounded-lg shadow p-6 mb-6">
                <div class="flex justify-between items-center mb-6">
                    <h2 class="text-xl font-bold text-gray-800"><?php echo htmlspecialchars($current_test['quiz_title']); ?></h2>
                    <div class="text-right">
                        <div id="timer" class="text-2xl font-bold text-red-600"><?php echo gmdate("i:s", $current_test['time_limit'] * 60); ?></div>
                        <p class="text-sm text-gray-600">Time Remaining</p>
                    </div>
                </div>
                
                <div class="mb-4 p-3 bg-blue-50 rounded-lg">
                    <p class="font-semibold">Instructions:</p>
                    <p><?php echo htmlspecialchars($current_test['instructions'] ?: 'Answer all questions to the best of your ability.'); ?></p>
                    <p id="progress" class="mt-2 font-medium text-blue-700">Progress: 0/<?php echo count($test_questions); ?> questions answered</p>
                </div>
                
                <!-- Question Navigation -->
                <div class="mb-6">
                    <h3 class="font-semibold mb-2">Question Navigation</h3>
                    <div class="question-navigation" id="question-nav">
                        <?php foreach ($test_questions as $index => $question): ?>
                            <div class="question-number" data-question="<?php echo $index; ?>" id="nav-<?php echo $index; ?>">
                                <?php echo $index + 1; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <form id="quizForm" method="POST" action="">
                    <input type="hidden" name="quiz_id" value="<?php echo $current_test['id']; ?>">
                    <input type="hidden" name="time_taken" id="time_taken" value="0">
                    
                    <div class="space-y-6">
                        <?php foreach ($test_questions as $index => $question): ?>
                            <div class="question-container p-4 border rounded-lg <?php echo $index === 0 ? 'active' : ''; ?>" id="question-<?php echo $index; ?>">
                                <p class="font-semibold mb-2">Question <?php echo $index + 1; ?>: 
                                    <span class="text-sm font-normal text-gray-600">(<?php echo $question['marks']; ?> marks)</span>
                                </p>
                                <p class="mb-3"><?php echo htmlspecialchars($question['question_text']); ?></p>
                                
                                <?php if ($question['question_type'] == 'multiple_choice' && !empty($question['options'])): ?>
                                    <div class="space-y-2 ml-4">
                                        <?php foreach ($question['options'] as $option): ?>
                                            <div class="flex items-center">
                                                <input type="radio" id="option_<?php echo $option['id']; ?>" 
                                                       name="answer_<?php echo $question['id']; ?>" 
                                                       value="<?php echo $option['id']; ?>" 
                                                       class="mr-2" onclick="updateProgress()" required>
                                                <label for="option_<?php echo $option['id']; ?>"><?php echo htmlspecialchars($option['option_text']); ?></label>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php elseif ($question['question_type'] == 'true_false'): ?>
                                    <div class="space-y-2 ml-4">
                                        <div class="flex items-center">
                                            <input type="radio" id="true_<?php echo $question['id']; ?>" 
                                                   name="answer_<?php echo $question['id']; ?>" 
                                                   value="true" class="mr-2" onclick="updateProgress()" required>
                                            <label for="true_<?php echo $question['id']; ?>">True</label>
                                        </div>
                                        <div class="flex items-center">
                                            <input type="radio" id="false_<?php echo $question['id']; ?>" 
                                                   name="answer_<?php echo $question['id']; ?>" 
                                                   value="false" class="mr-2" onclick="updateProgress()" required>
                                            <label for="false_<?php echo $question['id']; ?>">False</label>
                                        </div>
                                    </div>
                                <?php elseif ($question['question_type'] == 'short_answer'): ?>
                                    <div class="ml-4 mt-2">
                                        <textarea name="answer_<?php echo $question['id']; ?>" 
                                                  rows="3" class="w-full px-3 py-2 border rounded" 
                                                  placeholder="Type your answer here" oninput="updateProgress()" required></textarea>
                                    </div>
                                <?php elseif ($question['question_type'] == 'essay'): ?>
                                    <div class="ml-4 mt-2">
                                        <textarea name="answer_<?php echo $question['id']; ?>" 
                                                  rows="5" class="w-full px-3 py-2 border rounded" 
                                                  placeholder="Write your essay here" oninput="updateProgress()" required></textarea>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="mt-4 flex justify-between">
                                    <button type="button" class="bg-yellow-500 hover:bg-yellow-600 text-white px-3 py-1 rounded text-sm mark-question" data-question="<?php echo $index; ?>">
                                        <i class="fas fa-bookmark mr-1"></i> Mark for Review
                                    </button>
                                    <button type="button" class="bg-gray-500 hover:bg-gray-600 text-white px-3 py-1 rounded text-sm clear-question" data-question="<?php echo $index; ?>">
                                        <i class="fas fa-times mr-1"></i> Clear Answer
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <div class="mt-8 flex justify-between">
                        <button type="button" id="prevBtn" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded font-semibold hidden">
                            <i class="fas fa-arrow-left mr-2"></i> Previous
                        </button>
                        
                        <button type="button" id="nextBtn" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded font-semibold">
                            Next <i class="fas fa-arrow-right ml-2"></i>
                        </button>
                        
                        <button type="submit" name="submit_test" class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded font-semibold hidden" id="submitBtn">
                            <i class="fas fa-paper-plane mr-2"></i> Submit Test
                        </button>
                    </div>
                </form>
            </div>
                   
            <script>
                // Question navigation
                let currentQuestion = 0;
                const totalQuestions = <?php echo count($test_questions); ?>;
                const questionContainers = document.querySelectorAll('.question-container');
                const prevBtn = document.getElementById('prevBtn');
                const nextBtn = document.getElementById('nextBtn');
                const submitBtn = document.getElementById('submitBtn');
                const answeredQuestions = new Set();
                const markedQuestions = new Set();
                
                function showQuestion(index) {
                    // Hide all questions
                    questionContainers.forEach(container => {
                        container.classList.remove('active');
                    });
                    
                    // Show current question
                    questionContainers[index].classList.add('active');
                    
                    // Update navigation buttons
                    if (index === 0) {
                        prevBtn.classList.add('hidden');
                    } else {
                        prevBtn.classList.remove('hidden');
                    }
                    
                    if (index === totalQuestions - 1) {
                        nextBtn.classList.add('hidden');
                        submitBtn.classList.remove('hidden');
                    } else {
                        nextBtn.classList.remove('hidden');
                        submitBtn.classList.add('hidden');
                    }
                    
                    // Update navigation highlights
                    document.querySelectorAll('.question-number').forEach(nav => {
                        nav.classList.remove('current');
                    });
                    document.getElementById('nav-' + index).classList.add('current');
                    
                    // Update current question
                    currentQuestion = index;
                }
                
                // Navigation button event listeners
                prevBtn.addEventListener('click', () => {
                    if (currentQuestion > 0) {
                        currentQuestion--;
                        showQuestion(currentQuestion);
                    }
                });
                
                nextBtn.addEventListener('click', () => {
                    if (currentQuestion < totalQuestions - 1) {
                        currentQuestion++;
                        showQuestion(currentQuestion);
                    }
                });
                
                // Question navigation click
                document.querySelectorAll('.question-number').forEach(nav => {
                    nav.addEventListener('click', function() {
                        const questionIndex = parseInt(this.getAttribute('data-question'));
                        showQuestion(questionIndex);
                    });
                });
                
                // Mark question for review
                document.querySelectorAll('.mark-question').forEach(btn => {
                    btn.addEventListener('click', function() {
                        const questionIndex = parseInt(this.getAttribute('data-question'));
                        markedQuestions.add(questionIndex);
                        document.getElementById('nav-' + questionIndex).classList.add('marked');
                        this.innerHTML = '<i class="fas fa-check mr-1"></i> Marked';
                        this.classList.remove('bg-yellow-500', 'hover:bg-yellow-600');
                        this.classList.add('bg-green-500', 'hover:bg-green-600');
                    });
                });
                
                // Clear answer
                document.querySelectorAll('.clear-question').forEach(btn => {
                    btn.addEventListener('click', function() {
                        const questionIndex = parseInt(this.getAttribute('data-question'));
                        const questionId = <?php echo $test_questions[0]['id']; ?> + questionIndex;
                        
                        // Clear radio buttons
                        const radioButtons = document.querySelectorAll(`input[name="answer_${questionId}"]`);
                        radioButtons.forEach(radio => {
                            radio.checked = false;
                        });
                        
                        // Clear textareas
                        const textAreas = document.querySelectorAll(`textarea[name="answer_${questionId}"]`);
                        textAreas.forEach(textarea => {
                            textarea.value = '';
                        });
                        
                        // Update answered questions
                        answeredQuestions.delete(questionIndex);
                        document.getElementById('nav-' + questionIndex).classList.remove('answered');
                        
                        updateProgress();
                    });
                });
                
                // Start the timer when page loads
                document.addEventListener('DOMContentLoaded', function() {
                    const timeLimit = <?php echo $current_test['time_limit'] * 60; ?>;
                    const display = document.getElementById('timer');
                    const timeTakenInput = document.getElementById('time_taken');
                    let timeSpent = 0;
                    
                    const timerInterval = setInterval(function() {
                        timeSpent++;
                        timeTakenInput.value = timeSpent;
                    }, 1000);
                    
                    const countdown = startTimer(timeLimit, display);
                    
                    // Update current time every second
                    setInterval(function() {
                        const now = new Date();
                        document.getElementById('current-time').textContent = now.toLocaleString();
                    }, 1000);
                    
                    // Prevent leaving page during test
                    window.onbeforeunload = function() {
                        return "Are you sure you want to leave? Your progress may be lost.";
                    };
                    
                    // Remove the beforeunload handler when form is submitted
                    document.getElementById('quizForm').addEventListener('submit', function() {
                        window.onbeforeunload = null;
                    });
                    
                    // Initialize question navigation
                    showQuestion(0);
                    
                    // Add required attribute validation
                    document.querySelectorAll('input[type="radio"], textarea').forEach(element => {
                        element.addEventListener('change', function() {
                            this.setCustomValidity('');
                        });
                    });
                });
                
                function startTimer(duration, display) {
                    let timer = duration, minutes, seconds;
                    const interval = setInterval(function () {
                        minutes = parseInt(timer / 60, 10);
                        seconds = parseInt(timer % 60, 10);

                        minutes = minutes < 10 ? "0" + minutes : minutes;
                        seconds = seconds < 10 ? "0" + seconds : seconds;

                        display.textContent = minutes + ":" + seconds;

                        if (--timer < 0) {
                            clearInterval(interval);
                            alert("Time's up! Your test will be submitted automatically.");
                            document.getElementById('quizForm').submit();
                        }
                    }, 1000);
                    
                    return interval;
                }
                
                function updateProgress() {
                    const answered = document.querySelectorAll('input[type="radio"]:checked, textarea').length;
                    const progress = document.getElementById('progress');
                    
                    // Update answered questions set
                    document.querySelectorAll('input[type="radio"]:checked, textarea').forEach(element => {
                        if (element.value.trim() !== '') {
                            const name = element.name;
                            const questionId = name.replace('answer_', '');
                            
                            // Find the question index
                            <?php foreach ($test_questions as $index => $q): ?>
                                if (<?php echo $q['id']; ?> == questionId) {
                                    answeredQuestions.add(<?php echo $index; ?>);
                                    document.getElementById('nav-' + <?php echo $index; ?>).classList.add('answered');
                                }
                            <?php endforeach; ?>
                        }
                    });
                    
                    if (progress) {
                        progress.textContent = `Progress: ${answeredQuestions.size}/${totalQuestions} questions answered`;
                    }
                }
                
                // Validate form before submission
                document.getElementById('quizForm').addEventListener('submit', function(e) {
                    const unanswered = totalQuestions - answeredQuestions.size;
                    
                    if (unanswered > 0) {
                        if (!confirm(`You have ${unanswered} unanswered questions. Are you sure you want to submit?`)) {
                            e.preventDefault();
                            return false;
                        }
                    }
                    
                    return true;
                });
            </script>
            
        <!-- Test Results View -->
        <?php elseif ($test_result && $test_details): ?>
            <div class="bg-white rounded-lg shadow p-6 mb-6">
                <h2 class="text-xl font-bold text-gray-800 mb-4">Test Results: <?php echo htmlspecialchars($test_details['quiz_title']); ?></h2>
                
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                    <div class="bg-blue-50 p-4 rounded-lg text-center">
                        <p class="text-2xl font-bold"><?php echo $test_result['score'] . '/' . $test_result['total_marks']; ?></p>
                        <p class="text-gray-600">Score</p>
                    </div>
                    
                    <div class="bg-green-50 p-4 rounded-lg text-center">
                        <p class="text-2xl font-bold"><?php echo $test_result['percentage']; ?>%</p>
                        <p class="text-gray-600">Percentage</p>
                    </div>
                    
                    <div class="bg-purple-50 p-4 rounded-lg text-center">
                        <p class="text-2xl font-bold"><?php echo gmdate("i:s", $test_result['time_taken']); ?></p>
                        <p class="text-gray-600">Time Taken</p>
                    </div>
                </div>
                
                <div class="mt-6 bg-gray-50 p-4 rounded-lg">
                    <h3 class="font-semibold mb-2">Test Details:</h3>
                    <p><span class="font-medium">Subject:</span> <?php echo htmlspecialchars($test_details['subject_name']); ?></p>
                    <p><span class="font-medium">Submitted on:</span> <?php echo date('M j, Y g:i A', strtotime($test_result['submitted_at'])); ?></p>
                </div>
                
                <div class="mt-6 flex justify-center space-x-4">
                    <a href="cbttest.php" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Tests
                    </a>
                    
                    <button onclick="window.print()" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded">
                        <i class="fas fa-download mr-2"></i> Download Results
                    </button>
                </div>
            </div>

        <!-- Main Dashboard -->
        <?php else: ?>
            <!-- Available Tests -->
                      <!-- Available Tests -->
            <?php if (!empty($available_tests)): ?>
                <div class="bg-white rounded-lg shadow p-6 mb-6">
                    <h2 class="text-xl font-bold text-gray-800 mb-4">Available Tests</h2>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        <?php foreach ($available_tests as $test): ?>
                            <div class="border rounded-lg p-4 hover:shadow-lg transition-shadow">
                                <div class="flex justify-between items-start mb-3">
                                    <h3 class="font-semibold text-lg"><?php echo htmlspecialchars($test['quiz_title']); ?></h3>
                                    <span class="bg-green-100 text-green-800 text-xs font-medium px-2 py-1 rounded">Active</span>
                                </div>
                                
                                <p class="text-gray-600 mb-2"><?php echo htmlspecialchars($test['subject_name']); ?></p>
                                <p class="text-sm text-gray-500 mb-2"><?php echo $test['question_count']; ?> questions</p>
                                <p class="text-sm text-gray-500 mb-3">Time limit: <?php echo $test['time_limit']; ?> minutes</p>
                                
                                <?php if ($test['start_date'] || $test['end_date']): ?>
                                    <div class="text-xs text-gray-500 mb-3">
                                        <?php if ($test['start_date']): ?>
                                            <p>Started: <?php echo date('M j, Y g:i A', strtotime($test['start_date'])); ?></p>
                                        <?php endif; ?>
                                        <?php if ($test['end_date']): ?>
                                            <p>Ends: <?php echo date('M j, Y g:i A', strtotime($test['end_date'])); ?></p>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (empty($test['has_result'])): ?>
                                    <a href="cbttest.php?take_test=<?php echo $test['id']; ?>" class="block w-full bg-blue-600 hover:bg-blue-700 text-white text-center py-2 rounded">
                                        Start Test
                                    </a>
                                <?php else: ?>
                                    <div class="text-center py-2 bg-gray-100 text-gray-500 rounded">
                                        <i class="fas fa-check-circle mr-1"></i> Test Completed
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Upcoming Tests -->
                    <!-- Upcoming Tests -->
            <?php if (!empty($upcoming_tests)): ?>
                <div class="bg-white rounded-lg shadow p-6 mb-6">
                    <h2 class="text-xl font-bold text-gray-800 mb-4">Upcoming Tests</h2>
                    
                    <div class="overflow-x-auto">
                        <table class="min-w-full bg-white">
                            <thead class="bg-gray-200 text-gray-700">
                                <tr>
                                    <th class="py-3 px-4 text-left">Test Name</th>
                                    <th class="py-3 px-4 text-left">Subject</th>
                                    <th class="py-3 px-4 text-left">Starts In</th>
                                    <th class="py-3 px-4 text-left">Duration</th>
                                    <th class="py-3 px-4 text-left">Questions</th>
                                    <th class="py-3 px-4 text-left">Status</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-600">
                                <?php foreach ($upcoming_tests as $test): ?>
                                    <tr class="border-b border-gray-200 hover:bg-gray-50">
                                        <td class="py-3 px-4"><?php echo htmlspecialchars($test['quiz_title']); ?></td>
                                        <td class="py-3 px-4"><?php echo htmlspecialchars($test['subject_name']); ?></td>
                                        <td class="py-3 px-4">
                                            <span class="upcoming-test" 
                                                  data-start-time="<?php echo $test['start_date']; ?>"
                                                  data-test-id="<?php echo $test['id']; ?>">
                                                Calculating...
                                            </span>
                                        </td>
                                        <td class="py-3 px-4"><?php echo $test['time_limit']; ?> mins</td>
                                        <td class="py-3 px-4"><?php echo $test['question_count']; ?></td>
                                        <td class="py-3 px-4">
                                            <?php if (!empty($test['has_result'])): ?>
                                                <span class="text-green-600 font-medium">Completed</span>
                                            <?php else: ?>
                                                <span class="text-blue-600 font-medium">Upcoming</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Completed Tests -->
            <?php if (!empty($completed_tests)): ?>
                <div class="bg-white rounded-lg shadow p-6">
                    <h2 class="text-xl font-bold text-gray-800 mb-4">Test History</h2>
                    
                    <div class="overflow-x-auto">
                        <table class="min-w-full bg-white">
                            <thead class="bg-gray-200 text-gray-700">
                                <tr>
                                    <th class="py-3 px-4 text-left">Test Name</th>
                                    <th class="py-3 px-4 text-left">Subject</th>
                                    <th class="py-3 px-4 text-left">Taken On</th>
                                    <th class="py-3 px-4 text-left">Score</th>
                                    <th class="py-3 px-4 text-left">Status</th>
                                    <th class="py-3 px-4 text-left">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-600">
                                <?php foreach ($completed_tests as $test): ?>
                                    <tr class="border-b border-gray-200 hover:bg-gray-50">
                                        <td class="py-3 px-4"><?php echo htmlspecialchars($test['quiz_title']); ?></td>
                                        <td class="py-3 px-4"><?php echo htmlspecialchars($test['subject_name']); ?></td>
                                        <td class="py-3 px-4">
                                            <?php 
                                            $result_query = "SELECT submitted_at FROM quiz_results 
                                                            WHERE quiz_id = {$test['id']} AND student_id = $student_id";
                                            $result_result = mysqli_query($conn, $result_query);
                                            if ($result_result && mysqli_num_rows($result_result) > 0) {
                                                $result = mysqli_fetch_assoc($result_result);
                                                echo date('M j, Y g:i A', strtotime($result['submitted_at']));
                                            } else {
                                                echo 'N/A';
                                            }
                                            ?>
                                        </td>
                                        <td class="py-3 px-4">
                                            <?php 
                                            if ($test['can_view']) {
                                                $result_query = "SELECT score, total_marks FROM quiz_results 
                                                                WHERE quiz_id = {$test['id']} AND student_id = $student_id";
                                                $result_result = mysqli_query($conn, $result_query);
                                                if ($result_result && mysqli_num_rows($result_result) > 0) {
                                                    $result = mysqli_fetch_assoc($result_result);
                                                    echo $result['score'] . '/' . $result['total_marks'];
                                                } else {
                                                    echo 'N/A';
                                                }
                                            } else {
                                                echo 'Not released';
                                            }
                                            ?>
                                        </td>
                                        <td class="py-3 px-4">
                                            <span class="px-2 py-1 text-xs font-medium rounded-full 
                                                <?php echo $test['can_view'] ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                                <?php echo $test['can_view'] ? 'Results Available' : 'Pending Review'; ?>
                                            </span>
                                        </td>
                                        <td class="py-3 px-4">
                                            <?php if ($test['can_view']): ?>
                                                <a href="cbttest.php?view_result=<?php echo $test['id']; ?>" class="text-blue-600 hover:text-blue-800">
                                                    <i class="fas fa-chart-bar mr-1"></i> View Results
                                                </a>
                                            <?php else: ?>
                                                <span class="text-gray-400">
                                                    <i class="fas fa-clock mr-1"></i> Results Pending
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if (empty($available_tests) && empty($upcoming_tests) && empty($completed_tests)): ?>
                <div class="bg-white rounded-lg shadow p-6 text-center">
                    <i class="fas fa-clipboard-list text-4xl text-gray-400 mb-4"></i>
                    <h3 class="text-xl font-semibold text-gray-700 mb-2">No Tests Available</h3>
                    <p class="text-gray-500">You don't have any tests assigned at the moment. Please check back later.</p>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    
    <script>
        // Update current time every second
        setInterval(function() {
            const now = new Date();
            document.getElementById('current-time').textContent = now.toLocaleString();
        }, 1000);
        
        // Add countdown for upcoming tests
          // Add countdown for upcoming tests
        function updateUpcomingTestCountdowns() {
            document.querySelectorAll('.upcoming-test').forEach(element => {
                const startTime = element.getAttribute('data-start-time');
                const startTimestamp = new Date(startTime).getTime();
                const now = new Date().getTime();
                const timeLeft = startTimestamp - now;
                
                if (timeLeft > 0) {
                    const hours = Math.floor((timeLeft % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                    const minutes = Math.floor((timeLeft % (1000 * 60 * 60)) / (1000 * 60));
                    const seconds = Math.floor((timeLeft % (1000 * 60)) / 1000);
                    
                    element.innerHTML = `Starts in: ${hours}h ${minutes}m ${seconds}s`;
                    element.classList.add('test-countdown');
                } else {
                    // Check if test is already completed by looking at the row status
                    const row = element.closest('tr');
                    const statusCell = row.querySelector('td:last-child');
                    if (statusCell && statusCell.textContent.trim() === 'Completed') {
                        element.innerHTML = 'Test Completed';
                    } else {
                        element.innerHTML = 'Available now - <a href="cbttest.php?take_test=' + 
                                          element.getAttribute('data-test-id') + 
                                          '" class="text-blue-600 hover:underline">Start Test</a>';
                    }
                    element.classList.remove('test-countdown');
                }
            });
        }

        // Update countdown every second
        setInterval(updateUpcomingTestCountdowns, 1000);
        updateUpcomingTestCountdowns();
    </script>
</body>
</html>

<?php
ob_end_flush();
?>