<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require '../config/conn.php';

// Handle AJAX POST request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');

    if (!isset($_SESSION['user_id'])) {
        echo json_encode(['success' => false, 'message' => 'Not logged in']);
        exit;
    }

    $userId = $_SESSION['user_id'];
    $name   = trim($_POST['name'] ?? '');
    $email  = trim($_POST['email'] ?? '');
    $school = trim($_POST['school'] ?? '');
    $image  = '';

    // Fetch current image
    $stmt = $conn->prepare("SELECT image FROM users WHERE id=?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $image = $row['image'] ?? '';
    $stmt->close();

    // Handle image upload
    if (!empty($_FILES['profile_image']['name'])) {
        $targetDir = "../upload/";
        if (!file_exists($targetDir)) mkdir($targetDir, 0777, true);

        $fileName = time() . "_" . basename($_FILES["profile_image"]["name"]);
        $targetFilePath = $targetDir . $fileName;
        $fileType = strtolower(pathinfo($targetFilePath, PATHINFO_EXTENSION));

        $allowedTypes = ["jpg","jpeg","png","gif"];
        if (!in_array($fileType, $allowedTypes)) {
            echo json_encode(['success'=>false, 'message'=>'Only JPG, JPEG, PNG, GIF files allowed']);
            exit;
        }

        if (!move_uploaded_file($_FILES["profile_image"]["tmp_name"], $targetFilePath)) {
            echo json_encode(['success'=>false, 'message'=>'Failed to upload image']);
            exit;
        }

        $image = $fileName;
    }

    // Update user
    $update = $conn->prepare("UPDATE users SET name=?, email=?, school=?, image=? WHERE id=?");
    $update->bind_param("ssssi", $name, $email, $school, $image, $userId);
    if ($update->execute()) {
        echo json_encode(['success'=>true, 'message'=>'Profile updated successfully', 'image'=>$image]);
    } else {
        echo json_encode(['success'=>false, 'message'=>'Failed to update profile: '.$conn->error]);
    }
    exit;
}

// GET request – render HTML page
include 'include/header.php';

// Fetch user data
$userId = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT name,email,school,image FROM users WHERE id=?");
$stmt->bind_param("i", $userId);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();
?>

<div class="p-6 max-w-lg mx-auto">
  <h2 class="text-2xl font-bold mb-4">Edit Profile</h2>

  <form id="editProfileForm" enctype="multipart/form-data" class="space-y-4 bg-white p-6 rounded-lg shadow">
    <div class="flex items-center space-x-4">
      <img id="profilePreview" src="<?php echo !empty($user['image']) ? '../upload/'.$user['image'] : 'https://via.placeholder.com/100'; ?>" 
           alt="Profile" class="w-20 h-20 rounded-full border">
      <input type="file" name="profile_image" id="profileImage" class="text-sm text-gray-500">
    </div>
    <div>
      <label class="block text-sm font-medium">Full Name</label>
      <input type="text" name="name" value="<?php echo htmlspecialchars($user['name']); ?>" 
             class="w-full border rounded px-3 py-2" required>
    </div>
    <div>
      <label class="block text-sm font-medium">Email</label>
      <input type="email" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" 
             class="w-full border rounded px-3 py-2" required>
    </div>
    <div>
      <label class="block text-sm font-medium">School</label>
      <input type="text" name="school" value="<?php echo htmlspecialchars($user['school']); ?>" 
             class="w-full border rounded px-3 py-2">
    </div>
    <button type="submit" class="bg-primary text-white px-4 py-2 rounded hover:bg-primaryDark">
      Save Changes
    </button>
    <a href="profile.php" class="ml-2 text-gray-500">Cancel</a>
  </form>
</div>

<script>
const form = document.getElementById('editProfileForm');
const preview = document.getElementById('profilePreview');

document.getElementById('profileImage').addEventListener('change', function(e){
    const file = e.target.files[0];
    if(file){
        const reader = new FileReader();
        reader.onload = function(e){
            preview.src = e.target.result;
        }
        reader.readAsDataURL(file);
    }
});

form.addEventListener('submit', function(e){
    e.preventDefault();
    const formData = new FormData(form);

    fetch('edit-profile.php', {method:'POST', body:formData})
    .then(res => res.json())
    .then(data => {
        showToast(data.message, data.success ? 'success' : 'error');
        if(data.success && data.image){
            preview.src = '../upload/' + data.image;
        }
    })
    .catch(()=>showToast('Something went wrong. Please try again.', 'error'));
});

function showToast(message, type='success'){
    let toast = document.getElementById('toast');
    if(!toast){
        toast = document.createElement('div');
        toast.id = 'toast';
        toast.className = 'toast fixed top-4 right-4 px-4 py-2 rounded shadow text-white z-50';
        document.body.appendChild(toast);
    }
    toast.textContent = message;
    toast.className = 'toast fixed top-4 right-4 px-4 py-2 rounded shadow text-white z-50 ' + (type==='success'?'bg-green-500':'bg-red-500');
    toast.style.display = 'block';
    setTimeout(()=>{ toast.style.display='none'; }, 4000);
}
</script>

<?php include 'include/footer.php'; ?>
