<?php
include 'include/header.php';


// Check if user is logged in, if not redirect to login page
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type'])) {
    header('Location: ../portal_login.php');
    exit();
}

// Check if user is a student, if not redirect to appropriate dashboard
if ($_SESSION['user_type'] !== 'student') {
    if ($_SESSION['user_type'] === 'teacher') {
        header('Location: teacher_dashboard.php');
    } else if ($_SESSION['user_type'] === 'parent') {
        header('Location: parent_dashboard.php');
    }
    exit();
}

// Include database connection
include '../config/conn.php';

// Get student data including class and section
$student_id = $_SESSION['user_id'];
$query = "SELECT s.*, c.class_name, sec.section_name, c.id as class_id, sec.id as section_id
          FROM students s 
          LEFT JOIN classes c ON s.class = c.id 
          LEFT JOIN sections sec ON s.section = sec.id 
          WHERE s.id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("s", $student_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // Student not found in database, logout user
    session_destroy();
    header('Location: ../portal_login.php');
    exit();
}

$student = $result->fetch_assoc();
$stmt->close();

// Get the student's class and section IDs
$class_id = $student['class_id'];
$section_id = $student['section_id'];

// Get subjects for the student's class
$subjects_query = "SELECT s.id, s.subject_name, s.description, 
                  t.full_name as teacher_name, t.email as teacher_email
                  FROM subjects s
                  LEFT JOIN subject_teachers st ON s.id = st.subject_id
                  LEFT JOIN teachers t ON st.teacher_id = t.id
                  WHERE s.status = 'active'
                  ORDER BY s.subject_name";
$subjects_result = mysqli_query($conn, $subjects_query);

// Get timetable to see when each subject is scheduled
$timetable_query = "SELECT cs.subject_id, sub.subject_name, ts.period_name, 
                   ts.start_time, ts.end_time, cs.day_of_week, t.full_name as teacher_name
                   FROM class_schedules cs
                   JOIN subjects sub ON cs.subject_id = sub.id
                   JOIN time_slots ts ON cs.time_slot_id = ts.id
                   JOIN teachers t ON cs.teacher_id = t.id
                   WHERE cs.class_id = ? AND cs.section_id = ?
                   ORDER BY cs.day_of_week, ts.start_time";
$timetable_stmt = $conn->prepare($timetable_query);
$timetable_stmt->bind_param("ii", $class_id, $section_id);
$timetable_stmt->execute();
$timetable_result = $timetable_stmt->get_result();

// Organize timetable by subject
$subject_schedule = [];
while ($schedule = $timetable_result->fetch_assoc()) {
    $subject_id = $schedule['subject_id'];
    if (!isset($subject_schedule[$subject_id])) {
        $subject_schedule[$subject_id] = [
            'subject_name' => $schedule['subject_name'],
            'teacher_name' => $schedule['teacher_name'],
            'schedules' => []
        ];
    }
    $subject_schedule[$subject_id]['schedules'][] = $schedule;
}
$timetable_stmt->close();

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: ../portal_login.php');
    exit();
}

// Include header
?>

<div class="bg-white p-6 rounded-lg ">
    

    <!-- Student Information -->
    <div class="bg-blue-50 p-4 rounded-lg mb-6">
        <h3 class="text-lg font-semibold text-blue-800">
            <i class="fas fa-user-graduate mr-2"></i>
            <?php echo htmlspecialchars($student['full_name']); ?>
        </h3>
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mt-2">
            <div class="text-blue-600">
                <i class="fas fa-graduation-cap mr-1"></i> 
                Class: <?php echo htmlspecialchars($student['class_name']); ?>
            </div>
            <div class="text-blue-600">
                <i class="fas fa-layer-group mr-1"></i> 
                Section: <?php echo htmlspecialchars($student['section_name']); ?>
            </div>
            <div class="text-blue-600">
                <i class="fas fa-hashtag mr-1"></i> 
                Roll: <?php echo htmlspecialchars($student['roll_number']); ?>
            </div>
            <div class="text-blue-600">
                <i class="fas fa-book mr-1"></i> 
                Subjects: <?php echo mysqli_num_rows($subjects_result); ?>
            </div>
        </div>
    </div>

    <!-- Subjects Overview -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
        <?php 
        // Reset pointer and count subjects
        mysqli_data_seek($subjects_result, 0);
        $subject_count = mysqli_num_rows($subjects_result);
        $colors = ['bg-blue-500', 'bg-green-500', 'bg-purple-500', 'bg-orange-500', 
                  'bg-red-500', 'bg-yellow-500', 'bg-pink-500', 'bg-indigo-500'];
        $color_index = 0;
        
        while($subject = mysqli_fetch_assoc($subjects_result)): 
            $color = $colors[$color_index % count($colors)];
            $color_index++;
        ?>
            <div class="bg-white rounded-lg shadow-sm border-l-4 <?php echo $color; ?> p-4">
                <div class="flex items-center justify-between">
                    <h3 class="font-semibold text-gray-800"><?php echo htmlspecialchars($subject['subject_name']); ?></h3>
                    <div class="w-8 h-8 rounded-full <?php echo $color; ?> flex items-center justify-center text-white">
                        <?php echo strtoupper(substr($subject['subject_name'], 0, 1)); ?>
                    </div>
                </div>
                <p class="text-sm text-gray-600 mt-2">
                    <?php echo !empty($subject['description']) ? htmlspecialchars($subject['description']) : 'No description available'; ?>
                </p>
                <?php if (!empty($subject['teacher_name'])): ?>
                    <div class="mt-3 pt-2 border-t border-gray-100">
                        <p class="text-xs text-gray-500">Teacher:</p>
                        <p class="text-sm font-medium text-gray-700"><?php echo htmlspecialchars($subject['teacher_name']); ?></p>
                    </div>
                <?php endif; ?>
            </div>
        <?php endwhile; ?>
    </div>

    <!-- Detailed Subject Schedule -->
    <div class="bg-white p-6 rounded-lg shadow mb-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">
            <i class="fas fa-calendar-alt mr-2"></i> 
            Subject Schedule for <?php echo htmlspecialchars($student['class_name']); ?> - <?php echo htmlspecialchars($student['section_name']); ?>
        </h3>
        
        <?php if (!empty($subject_schedule)): ?>
            <div class="space-y-4">
                <?php foreach ($subject_schedule as $subject_id => $subject_data): ?>
                    <div class="border rounded-lg p-4">
                        <div class="flex justify-between items-start mb-3">
                            <h4 class="text-lg font-semibold text-gray-800">
                                <i class="fas fa-book mr-2"></i>
                                <?php echo htmlspecialchars($subject_data['subject_name']); ?>
                            </h4>
                            <?php if (!empty($subject_data['teacher_name'])): ?>
                                <span class="bg-blue-100 text-blue-800 text-xs font-medium px-2.5 py-0.5 rounded-full">
                                    <i class="fas fa-chalkboard-teacher mr-1"></i>
                                    <?php echo htmlspecialchars($subject_data['teacher_name']); ?>
                                </span>
                            <?php endif; ?>
                        </div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                            <?php foreach ($subject_data['schedules'] as $schedule): 
                                $start_time = date('h:i A', strtotime($schedule['start_time']));
                                $end_time = date('h:i A', strtotime($schedule['end_time']));
                            ?>
                                <div class="bg-gray-50 p-3 rounded-lg">
                                    <div class="flex items-center justify-between mb-2">
                                        <span class="text-sm font-medium text-gray-700">
                                            <i class="fas fa-calendar-day mr-1"></i>
                                            <?php echo htmlspecialchars($schedule['day_of_week']); ?>
                                        </span>
                                        <span class="text-xs bg-green-100 text-green-800 px-2 py-1 rounded">
                                            <?php echo !empty($schedule['period_name']) ? htmlspecialchars($schedule['period_name']) : 'Class'; ?>
                                        </span>
                                    </div>
                                    <div class="text-sm text-gray-600">
                                        <i class="fas fa-clock mr-1"></i>
                                        <?php echo $start_time . ' - ' . $end_time; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="text-center py-8 text-gray-500">
                <i class="fas fa-calendar-times text-4xl mb-3"></i>
                <p>No schedule information available for your subjects.</p>
                <p class="text-sm mt-2">Please check back later or contact your administrator.</p>
            </div>
        <?php endif; ?>
    </div>

    <!-- All Subjects List -->
    <div class="bg-white p-6 rounded-lg shadow">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">
            <i class="fas fa-list-alt mr-2"></i> 
            All Subjects in <?php echo htmlspecialchars($student['class_name']); ?>
        </h3>
        
        <?php 
        // Reset pointer for subjects result
        mysqli_data_seek($subjects_result, 0);
        if (mysqli_num_rows($subjects_result) > 0): 
        ?>
            <div class="overflow-x-auto">
                <table class="min-w-full bg-white">
                    <thead class="bg-gray-200 text-gray-700">
                        <tr>
                            <th class="py-3 px-4 text-left">Subject Name</th>
                            <th class="py-3 px-4 text-left">Description</th>
                            <th class="py-3 px-4 text-left">Teacher</th>
                            <th class="py-3 px-4 text-left">Classes per Week</th>
                        </tr>
                    </thead>
                    <tbody class="text-gray-600">
                        <?php while($subject = mysqli_fetch_assoc($subjects_result)): 
                            $class_count = 0;
                            if (isset($subject_schedule[$subject['id']])) {
                                $class_count = count($subject_schedule[$subject['id']]['schedules']);
                            }
                        ?>
                            <tr class="border-b border-gray-200 hover:bg-gray-50">
                                <td class="py-3 px-4 font-medium">
                                    <i class="fas fa-book text-blue-500 mr-2"></i>
                                    <?php echo htmlspecialchars($subject['subject_name']); ?>
                                </td>
                                <td class="py-3 px-4">
                                    <?php echo !empty($subject['description']) ? htmlspecialchars($subject['description']) : 'No description'; ?>
                                </td>
                                <td class="py-3 px-4">
                                    <?php if (!empty($subject['teacher_name'])): ?>
                                        <div class="flex items-center">
                                            <i class="fas fa-chalkboard-teacher text-green-500 mr-2"></i>
                                            <?php echo htmlspecialchars($subject['teacher_name']); ?>
                                        </div>
                                    <?php else: ?>
                                        <span class="text-gray-400">Not assigned</span>
                                    <?php endif; ?>
                                </td>
                                <td class="py-3 px-4">
                                    <span class="bg-blue-100 text-blue-800 text-xs font-medium px-2.5 py-0.5 rounded-full">
                                        <?php echo $class_count; ?> class<?php echo $class_count !== 1 ? 'es' : ''; ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="text-center py-8 text-gray-500">
                <i class="fas fa-book-open text-4xl mb-3"></i>
                <p>No subjects found for your class.</p>
                <p class="text-sm mt-2">Please contact your administrator if this seems incorrect.</p>
            </div>
        <?php endif; ?>
    </div>

    <!-- Subject Statistics -->
    <div class="mt-6 bg-gray-50 p-6 rounded-lg">
        <h4 class="text-md font-medium text-gray-700 mb-4">
            <i class="fas fa-chart-bar mr-2"></i> Subject Statistics
        </h4>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div class="bg-white p-4 rounded-lg shadow text-center">
                <div class="text-2xl font-bold text-blue-600"><?php echo $subject_count; ?></div>
                <div class="text-sm text-gray-600">Total Subjects</div>
            </div>
            
            <div class="bg-white p-4 rounded-lg shadow text-center">
                <div class="text-2xl font-bold text-green-600">
                    <?php 
                    $teachers_count = 0;
                    $unique_teachers = [];
                    mysqli_data_seek($subjects_result, 0);
                    while($subject = mysqli_fetch_assoc($subjects_result)) {
                        if (!empty($subject['teacher_name']) && !in_array($subject['teacher_name'], $unique_teachers)) {
                            $unique_teachers[] = $subject['teacher_name'];
                            $teachers_count++;
                        }
                    }
                    echo $teachers_count;
                    ?>
                </div>
                <div class="text-sm text-gray-600">Different Teachers</div>
            </div>
            
            <div class="bg-white p-4 rounded-lg shadow text-center">
                <div class="text-2xl font-bold text-purple-600">
                    <?php 
                    $total_classes = 0;
                    foreach ($subject_schedule as $subject_data) {
                        $total_classes += count($subject_data['schedules']);
                    }
                    echo $total_classes;
                    ?>
                </div>
                <div class="text-sm text-gray-600">Weekly Classes</div>
            </div>
        </div>
    </div>
</div>

<script>
// Add interactive features
document.addEventListener('DOMContentLoaded', function() {
    // Add click events to subject cards to show more details
    const subjectCards = document.querySelectorAll('.bg-white.rounded-lg.shadow-sm');
    subjectCards.forEach(card => {
        card.addEventListener('click', function() {
            this.classList.toggle('bg-blue-50');
            this.classList.toggle('border-blue-300');
        });
    });
    
    // Highlight current day in the schedule
    const today = new Date().toLocaleDateString('en-US', { weekday: 'long' });
    const scheduleDays = document.querySelectorAll('.bg-gray-50.p-3.rounded-lg');
    
    scheduleDays.forEach(day => {
        const dayElement = day.querySelector('.text-sm.font-medium.text-gray-700');
        if (dayElement && dayElement.textContent.includes(today)) {
            day.classList.add('bg-green-50', 'border', 'border-green-200');
        }
    });
    
    // Add search functionality to the subjects table
    const searchInput = document.createElement('input');
    searchInput.type = 'text';
    searchInput.placeholder = 'Search subjects...';
    searchInput.className = 'w-full px-4 py-2 border rounded-lg mb-4 focus:outline-none focus:ring-2 focus:ring-blue-500';
    
    const subjectsTable = document.querySelector('table');
    if (subjectsTable) {
        subjectsTable.parentNode.insertBefore(searchInput, subjectsTable);
        
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = subjectsTable.querySelectorAll('tbody tr');
            
            rows.forEach(row => {
                const subjectName = row.querySelector('td:first-child').textContent.toLowerCase();
                const teacherName = row.querySelector('td:nth-child(3)').textContent.toLowerCase();
                
                if (subjectName.includes(searchTerm) || teacherName.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
    }
});
</script>

<?php include 'include/footer.php'; ?>