const CACHE_NAME = "adinoheight-cache-v2";
const ASSETS = [
  "/",
  "/index.php",
  "/parent/",
  "/parent/index.php",
  "/parent/notifications.php",
  "/css/styles.css",
  "/js/app.js",
  "/img/logotwo.png",
  "/offline.html"
];

// Install: cache files
self.addEventListener("install", (event) => {
  console.log("Service Worker installing...");
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then((cache) => {
        console.log("Caching app shell");
        return cache.addAll(ASSETS);
      })
      .then(() => self.skipWaiting())
  );
});

// Activate: clear old caches
self.addEventListener("activate", (event) => {
  console.log("Service Worker activating...");
  event.waitUntil(
    caches.keys().then((keys) =>
      Promise.all(
        keys.map((key) => {
          if (key !== CACHE_NAME) {
            console.log("Deleting old cache:", key);
            return caches.delete(key);
          }
        })
      )
    ).then(() => self.clients.claim())
  );
});

// Fetch: serve from cache, else network, else fallback
self.addEventListener("fetch", (event) => {
  // Skip cross-origin requests
  if (!event.request.url.startsWith(self.location.origin)) {
    return;
  }

  // Skip non-GET requests
  if (event.request.method !== "GET") {
    return;
  }

  event.respondWith(
    caches.match(event.request)
      .then((response) => {
        // Return cached version or fetch from network
        return response || fetch(event.request)
          .then((fetchResponse) => {
            // Cache the response for future use
            return caches.open(CACHE_NAME)
              .then((cache) => {
                cache.put(event.request, fetchResponse.clone());
                return fetchResponse;
              });
          })
          .catch(() => {
            // If both cache and network fail, show offline page
            if (event.request.destination === "document") {
              return caches.match("/offline.html");
            }
          });
      })
  );
});

// Push Notifications
self.addEventListener("push", (event) => {
  console.log("Push notification received", event);
  
  if (!event.data) {
    console.log("Push event but no data");
    return;
  }

  let data = {};
  try {
    data = event.data.json();
  } catch (e) {
    console.log("Push data is not JSON, using text");
    data = { 
      title: "Adinoheightschool", 
      body: event.data.text() || "New notification from school",
      icon: "/img/logotwo.png"
    };
  }

  const options = {
    body: data.body || "New notification from Adinoheightschool",
    icon: data.icon || "/img/logotwo.png",
    badge: "/img/logotwo.png",
    tag: data.tag || "parent-notification",
    data: data.data || { url: "/parent/notifications.php" },
    actions: [
      {
        action: 'view',
        title: 'View Notifications'
      },
      {
        action: 'dismiss',
        title: 'Dismiss'
      }
    ],
    vibrate: [200, 100, 200],
    requireInteraction: true
  };

  event.waitUntil(
    self.registration.showNotification(data.title || "Adinoheightschool", options)
  );
});

// Notification Click
self.addEventListener("notificationclick", (event) => {
  console.log("Notification clicked", event);
  
  event.notification.close();
  
  const urlToOpen = event.notification.data.url || "/parent/notifications.php";

  event.waitUntil(
    clients.matchAll({ 
      type: "window",
      includeUncontrolled: true 
    })
    .then((windowClients) => {
      // Check if there's already a window/tab open with the target URL
      for (let client of windowClients) {
        if (client.url.includes('/parent/') && "focus" in client) {
          return client.focus();
        }
      }
      
      // If no window is open, open a new one
      if (clients.openWindow) {
        return clients.openWindow(urlToOpen);
      }
    })
  );
});

// Notification Close
self.addEventListener("notificationclose", (event) => {
  console.log("Notification closed", event);
});

// Background Sync for offline attendance
self.addEventListener("sync", (event) => {
  console.log("Background sync triggered:", event.tag);
  
  if (event.tag === "background-attendance-sync") {
    event.waitUntil(doBackgroundSync());
  }
});

// Background Sync implementation
async function doBackgroundSync() {
  try {
    console.log("Starting background sync...");
    
    // Open IndexedDB for offline data
    const db = await openAttendanceDB();
    const pending = await getPendingAttendance(db);

    console.log(`Found ${pending.length} pending attendance records`);

    for (const record of pending) {
      try {
        const response = await fetch("/submit_attendance.php", {
          method: "POST",
          headers: { 
            "Content-Type": "application/json",
            "X-Offline-Sync": "true"
          },
          body: JSON.stringify(record.data),
        });

        if (response.ok) {
          await markAsSynced(db, record.id);
          console.log("Successfully synced record:", record.id);
        } else {
          console.error("Sync failed for record:", record.id, response.status);
        }
      } catch (error) {
        console.error("Error syncing record:", record.id, error);
      }
    }
  } catch (error) {
    console.error("Background sync error:", error);
  }
}

// IndexedDB helper functions
function openAttendanceDB() {
  return new Promise((resolve, reject) => {
    const request = indexedDB.open("AttendanceDB", 1);
    
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve(request.result);
    
    request.onupgradeneeded = (event) => {
      const db = event.target.result;
      if (!db.objectStoreNames.contains("pendingAttendance")) {
        const store = db.createObjectStore("pendingAttendance", { 
          keyPath: "id", 
          autoIncrement: true 
        });
        store.createIndex("timestamp", "timestamp", { unique: false });
      }
    };
  });
}

function getPendingAttendance(db) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction(["pendingAttendance"], "readonly");
    const store = transaction.objectStore("pendingAttendance");
    const request = store.getAll();
    
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve(request.result);
  });
}

function markAsSynced(db, id) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction(["pendingAttendance"], "readwrite");
    const store = transaction.objectStore("pendingAttendance");
    const request = store.delete(id);
    
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve();
  });
}

// Periodic sync for notifications
self.addEventListener("periodicsync", (event) => {
  if (event.tag === "notification-sync") {
    event.waitUntil(syncNotifications());
  }
});

async function syncNotifications() {
  try {
    const response = await fetch("/parent/get_notifications.php");
    const notifications = await response.json();
    
    // Show notifications for unread items
    notifications.forEach(notification => {
      if (!notification.is_read) {
        self.registration.showNotification(notification.title, {
          body: notification.body,
          icon: "/img/logotwo.png",
          tag: "notification-" + notification.id,
          data: { url: "/parent/notifications.php" }
        });
      }
    });
  } catch (error) {
    console.error("Notification sync error:", error);
  }
}