<?php
ob_start();
include 'include/header.php';
include '../config/conn.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
    header('Location: login.php');
    exit();
} // <-- This closing brace was missing

$teacher_id = $_SESSION['user_id'];
$success = '';
$error = '';

// Initialize variables
$class_id = isset($_GET['class_id']) ? intval($_GET['class_id']) : 0;
$section_id = isset($_GET['section_id']) ? intval($_GET['section_id']) : 0;
$attendance_date = isset($_GET['attendance_date']) ? $_GET['attendance_date'] : date('Y-m-d');
$month = isset($_GET['month']) ? $_GET['month'] : date('Y-m');
$active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'marking';

// Get teacher's assigned classes and sections
$teacher_assignments = [];
$assignment_query = "
    (SELECT id, class_name as name, 'class' as type, id as class_id, NULL as section_id 
     FROM classes 
     WHERE class_teacher_id = ? AND status = 'active')
    UNION
    (SELECT s.id, s.section_name as name, 'section' as type, s.class_id, s.id as section_id 
     FROM sections s 
     WHERE s.section_teacher_id = ? AND s.status = 'active')
";

$assignment_stmt = $conn->prepare($assignment_query);
$assignment_stmt->bind_param("ii", $teacher_id, $teacher_id);
$assignment_stmt->execute();
$assignment_result = $assignment_stmt->get_result();

while ($assignment = $assignment_result->fetch_assoc()) {
    $teacher_assignments[] = $assignment;
}
$assignment_stmt->close();

// Extract classes and sections from assignments
$teacher_classes = [];
$teacher_sections = [];

foreach ($teacher_assignments as $assignment) {
    if ($assignment['type'] === 'class') {
        $teacher_classes[] = [
            'id' => $assignment['id'],
            'class_name' => $assignment['name']
        ];
    } else {
        $teacher_sections[] = [
            'id' => $assignment['section_id'],
            'section_name' => $assignment['name'],
            'class_id' => $assignment['class_id']
        ];
    }
}

// Verify teacher has access to the selected class/section
$has_access = false;
if ($class_id > 0) {
    foreach ($teacher_classes as $class) {
        if ($class['id'] == $class_id) {
            $has_access = true;
            break;
        }
    }
    
    if (!$has_access && $section_id > 0) {
        foreach ($teacher_sections as $section) {
            if ($section['id'] == $section_id && $section['class_id'] == $class_id) {
                $has_access = true;
                break;
            }
        }
    }
}

// If no access to selected class/section, reset to first available option
if (!$has_access && (count($teacher_classes) > 0 || count($teacher_sections) > 0)) {
    if (count($teacher_classes) > 0) {
        $class_id = $teacher_classes[0]['id'];
        foreach ($teacher_sections as $section) {
            if ($section['class_id'] == $class_id) {
                $section_id = $section['id'];
                break;
            }
        }
    } else if (count($teacher_sections) > 0) {
        $section_id = $teacher_sections[0]['id'];
        $class_id = $teacher_sections[0]['class_id'];
    }
    $has_access = true;
}

// Fetch sections based on selected class
$sections = [];
if ($class_id > 0 && $has_access) {
    $sections_query = "SELECT s.id, s.section_name 
                      FROM sections s 
                      WHERE s.class_id = ? AND s.status = 'active'
                      AND (s.section_teacher_id = ? OR ? IN (
                          SELECT class_teacher_id FROM classes WHERE id = ? AND status = 'active'
                      ))";
    $sections_stmt = $conn->prepare($sections_query);
    $sections_stmt->bind_param("iiii", $class_id, $teacher_id, $teacher_id, $class_id);
    $sections_stmt->execute();
    $sections_result = $sections_stmt->get_result();
    
    while ($section = $sections_result->fetch_assoc()) {
        $sections[] = $section;
    }
    $sections_stmt->close();
}

// Get class and section names for display
$class_name = '';
$section_name = '';
if ($class_id > 0 && $has_access) {
    $class_name_query = "SELECT class_name FROM classes WHERE id = ?";
    $class_name_stmt = $conn->prepare($class_name_query);
    $class_name_stmt->bind_param("i", $class_id);
    $class_name_stmt->execute();
    $class_name_result = $class_name_stmt->get_result();
    
    if ($class_name_result && $class_name_result->num_rows > 0) {
        $class_name = $class_name_result->fetch_assoc()['class_name'];
    }
    $class_name_stmt->close();
}

if ($section_id > 0 && $has_access) {
    $section_name_query = "SELECT section_name FROM sections WHERE id = ?";
    $section_name_stmt = $conn->prepare($section_name_query);
    $section_name_stmt->bind_param("i", $section_id);
    $section_name_stmt->execute();
    $section_name_result = $section_name_stmt->get_result();
    
    if ($section_name_result && $section_name_result->num_rows > 0) {
        $section_name = $section_name_result->fetch_assoc()['section_name'];
    }
    $section_name_stmt->close();
}

// ========== ATTENDANCE MARKING FUNCTIONALITY ==========
// Fetch students based on selected class and section (only if teacher has access)
$students = [];
if ($has_access && $class_id > 0 && $section_id > 0) {
    $students_query = "SELECT id, student_id, full_name, roll_number 
                       FROM students 
                       WHERE class = ? AND section = ? AND status = 'active' 
                       ORDER BY roll_number";
    $students_stmt = $conn->prepare($students_query);
    $students_stmt->bind_param("ii", $class_id, $section_id);
    $students_stmt->execute();
    $students_result = $students_stmt->get_result();
    
    while ($student = $students_result->fetch_assoc()) {
        $students[] = $student;
    }
    $students_stmt->close();
}

// Check if attendance already exists for the selected date, class, and section
$attendance_exists = false;
$existing_attendance = [];
if ($has_access && $class_id > 0 && $section_id > 0 && !empty($attendance_date)) {
    $check_attendance_query = "SELECT student_id, status 
                               FROM attendance 
                               WHERE class_id = ? 
                               AND section_id = ? 
                               AND attendance_date = ?";
    $check_attendance_stmt = $conn->prepare($check_attendance_query);
    $check_attendance_stmt->bind_param("iis", $class_id, $section_id, $attendance_date);
    $check_attendance_stmt->execute();
    $check_attendance_result = $check_attendance_stmt->get_result();
    
    if ($check_attendance_result->num_rows > 0) {
        $attendance_exists = true;
        while ($row = $check_attendance_result->fetch_assoc()) {
            $existing_attendance[$row['student_id']] = $row['status'];
        }
    }
    $check_attendance_stmt->close();
}

// Handle attendance submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_attendance']) && $has_access) {
    $class_id = intval($_POST['class_id']);
    $section_id = intval($_POST['section_id']);
    $attendance_date = $_POST['attendance_date'];
    $attendance_data = $_POST['attendance'];
    
    // Validate inputs
    if (empty($class_id) || empty($section_id) || empty($attendance_date)) {
        $error = "Please select class, section, and date.";
    } else {
        // Verify the teacher still has access to this class/section
        $access_verified = false;
        
        // Check if teacher is class teacher
        foreach ($teacher_classes as $class) {
            if ($class['id'] == $class_id) {
                $access_verified = true;
                break;
            }
        }
        
        // If not class teacher, check if section teacher
        if (!$access_verified) {
            foreach ($teacher_sections as $section) {
                if ($section['id'] == $section_id && $section['class_id'] == $class_id) {
                    $access_verified = true;
                    break;
                }
            }
        }
        
        if (!$access_verified) {
            $error = "You don't have permission to mark attendance for this class/section.";
        } else {
            // Verify the teacher exists in users table with teacher role
            $verify_teacher_query = "SELECT id FROM users WHERE id = ? AND role = 'teacher' AND status = 'active'";
            $verify_stmt = $conn->prepare($verify_teacher_query);
            $verify_stmt->bind_param("i", $teacher_id);
            $verify_stmt->execute();
            $verify_result = $verify_stmt->get_result();
            
            if ($verify_result->num_rows === 0) {
                $error = "Invalid teacher account. Please contact administrator.";
            } else {
                // Begin transaction
                $conn->begin_transaction();
                
                try {
                    // Use INSERT...ON DUPLICATE KEY UPDATE to handle duplicates gracefully
                    foreach ($attendance_data as $student_id => $status) {
                        $student_id = intval($student_id);
                        $status = $conn->real_escape_string($status);
                        
                        $insert_query = "INSERT INTO attendance (student_id, class_id, section_id, attendance_date, status, recorded_by, recorded_at) 
                                         VALUES ($student_id, $class_id, $section_id, '$attendance_date', '$status', $teacher_id, NOW())
                                         ON DUPLICATE KEY UPDATE 
                                         status = '$status', 
                                         recorded_by = $teacher_id, 
                                         recorded_at = NOW()";
                        
                        if (!$conn->query($insert_query)) {
                            throw new Exception("Failed to update attendance for student ID: $student_id");
                        }
                    }
                    
                    // Commit transaction
                    $conn->commit();
                    $success = "Attendance recorded successfully!";
                    
                    // Refresh existing attendance data
                    $attendance_exists = true;
                    $existing_attendance = [];
                    foreach ($attendance_data as $student_id => $status) {
                        $existing_attendance[$student_id] = $status;
                    }
                    
                } catch (Exception $e) {
                    // Rollback transaction on error
                    $conn->rollback();
                    $error = "Error recording attendance: " . $e->getMessage();
                }
            }
            $verify_stmt->close();
        }
    }
}

// ========== CALENDAR FUNCTIONALITY ==========
// Get calendar data for the selected month
$first_day = date('N', strtotime($month . '-01'));
$days_in_month = date('t', strtotime($month . '-01'));
$month_name = date('F Y', strtotime($month . '-01'));

// Get attendance summary for the selected class/section and month
$attendance_summary = [];
if ($has_access && $class_id > 0 && $section_id > 0) {
    $start_date = date('Y-m-01', strtotime($month));
    $end_date = date('Y-m-t', strtotime($month));
    
    $summary_query = "
        SELECT 
            a.attendance_date,
            COUNT(*) as total_students,
            SUM(CASE WHEN a.status = 'present' THEN 1 ELSE 0 END) as present_count,
            SUM(CASE WHEN a.status = 'absent' THEN 1 ELSE 0 END) as absent_count,
            SUM(CASE WHEN a.status = 'late' THEN 1 ELSE 0 END) as late_count,
            SUM(CASE WHEN a.status = 'half_day' THEN 1 ELSE 0 END) as half_day_count
        FROM attendance a
        JOIN students s ON a.student_id = s.id
        WHERE a.class_id = ? 
        AND a.section_id = ?
        AND a.attendance_date BETWEEN ? AND ?
        GROUP BY a.attendance_date
        ORDER BY a.attendance_date
    ";
    
    $summary_stmt = $conn->prepare($summary_query);
    $summary_stmt->bind_param("iiss", $class_id, $section_id, $start_date, $end_date);
    $summary_stmt->execute();
    $summary_result = $summary_stmt->get_result();
    
    while ($row = $summary_result->fetch_assoc()) {
        $attendance_summary[$row['attendance_date']] = $row;
    }
    $summary_stmt->close();
}

// Get detailed attendance for a specific date if requested
$detailed_date = isset($_GET['date']) ? $_GET['date'] : '';
$detailed_attendance = [];
if (!empty($detailed_date) && $has_access && $class_id > 0 && $section_id > 0) {
    $detail_query = "
        SELECT a.student_id, a.status, s.full_name, s.roll_number
        FROM attendance a
        JOIN students s ON a.student_id = s.id
        WHERE a.class_id = ? 
        AND a.section_id = ?
        AND a.attendance_date = ?
        ORDER BY s.roll_number
    ";
    
    $detail_stmt = $conn->prepare($detail_query);
    $detail_stmt->bind_param("iis", $class_id, $section_id, $detailed_date);
    $detail_stmt->execute();
    $detail_result = $detail_stmt->get_result();
    
    while ($row = $detail_result->fetch_assoc()) {
        $detailed_attendance[] = $row;
    }
    $detail_stmt->close();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Attendance System</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .attendance-option {
            transition: all 0.2s ease;
        }
        .attendance-option:hover {
            transform: scale(1.05);
        }
        .attendance-option input:checked + span {
            font-weight: bold;
        }
        .calendar-day {
            transition: all 0.2s ease;
            min-height: 80px;
        }
        .calendar-day:hover {
            transform: scale(1.02);
           
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
        .tab-button {
            transition: all 0.3s ease;
        }
        .tab-button.active {
            background-color: #3b82f6;
            color: white;
        }
    </style>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-2 py-2">
        <div class="bg-white p-6 ">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-2xl font-bold text-gray-800">Student Attendance System</h2>
                <a href="dashboard.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 ">
                    <i class="fas fa-arrow-left mr-2"></i> Back to Dashboard
                </a>
            </div>

            <?php if (!empty($success)): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3  mb-4">
                    <?php echo $success; ?>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($error)): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3  mb-4">
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <!-- Tab Navigation -->
            <div class="flex border-b mb-6">
                <button class="tab-button py-3 px-6 font-medium mr-2 <?php echo $active_tab === 'marking' ? 'active' : 'bg-gray-200 text-gray-700'; ?>" 
                        onclick="switchTab('marking')">
                    <i class="fas fa-edit mr-2"></i> Mark Attendance
                </button>
                <button class="tab-button py-3 px-6 font-medium <?php echo $active_tab === 'calendar' ? 'active' : 'bg-gray-200 text-gray-700'; ?>" 
                        onclick="switchTab('calendar')">
                    <i class="fas fa-calendar-alt mr-2"></i> View Calendar
                </button>
        <?php if ($class_id && $section_id && $attendance_date): ?>
    <a href="export_attendance.php?class_id=<?php echo $class_id; ?>&section_id=<?php echo $section_id; ?>&attendance_date=<?php echo $attendance_date; ?>" 
       class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded inline-flex items-center">
        <i class="fa fa-file-excel mr-2"></i> Export Attendance
    </a>
<?php endif; ?>
            </div>

            <!-- Filter Section -->
            <div class="mb-6 bg-gray-50 p-2 ">
                <form method="GET" action="" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <input type="hidden" name="tab" id="tab_field" value="<?php echo $active_tab; ?>">
                    
                    <!-- Class Filter -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Select Class</label>
                        <select name="class_id" id="class_filter" class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500" >
                            <option value="">Select Class</option>
                            <?php foreach ($teacher_classes as $class): ?>
                                <option value="<?php echo $class['id']; ?>" <?php echo ($class_id == $class['id']) ? 'selected' : ''; ?>>
                                    <?php echo $class['class_name']; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Section Filter -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Select Section</label>
                        <select name="section_id" id="section_filter" class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500" <?php echo ($class_id == 0) ? 'disabled' : ''; ?> required>
                            <option value="">Select Section</option>
                            <?php foreach ($sections as $section): ?>
                                <option value="<?php echo $section['id']; ?>" <?php echo ($section_id == $section['id']) ? 'selected' : ''; ?>>
                                    <?php echo $section['section_name']; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Date/Month Selection based on active tab -->
                    <?php if ($active_tab === 'marking'): ?>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Select Date</label>
                            <input type="date" name="attendance_date" value="<?php echo $attendance_date; ?>" 
                                class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                        </div>
                    <?php else: ?>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Select Month</label>
                            <input type="month" name="month" value="<?php echo $month; ?>" 
                                class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Buttons -->
                    <div class="flex items-end space-x-2">
                        <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2  flex items-center h-10">
                            <i class="fas fa-filter mr-2"></i> Load
                        </button>
                        <a href="attendance.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2  flex items-center h-10">
                            <i class="fas fa-times mr-2"></i> Clear
                        </a>
                

                    </div>
                </form>
            </div>

            <!-- Mark Attendance Tab -->
            <div id="marking-tab" class="tab-content <?php echo $active_tab === 'marking' ? 'active' : ''; ?>">
                <?php if ($has_access && $class_id > 0 && $section_id > 0): ?>
                    <!-- Attendance Form -->
                    <div class="mb-6 bg-white p-2  ">
                        <div class="flex justify-between items-center mb-4">
                            <h3 class="text-lg font-semibold text-gray-800">
                                Mark Attendance for 
                                <?php 
                                echo $class_name . " - " . $section_name . " (" . date('d M Y', strtotime($attendance_date)) . ")";
                                ?>
                            </h3>
                            
                            <?php if ($attendance_exists): ?>
                                <span class="bg-blue-100 text-blue-800 text-xs font-medium px-2.5 py-0.5 -full">
                                    <i class="fas fa-info-circle mr-1"></i> Attendance already recorded for this date
                                </span>
                            <?php endif; ?>
                        </div>
                        
                        <?php if (!empty($students)): ?>
                            <form method="POST" action="">
                                <input type="hidden" name="class_id" value="<?php echo $class_id; ?>">
                                <input type="hidden" name="section_id" value="<?php echo $section_id; ?>">
                                <input type="hidden" name="attendance_date" value="<?php echo $attendance_date; ?>">
                                
                                <div class="overflow-x-auto">
                                    <table class="min-w-full bg-white">
                                        <thead class="bg-gray-200 text-gray-700">
                                            <tr>
                                                <th class="py-3 px-4 text-left">Roll No</th>
                                                <th class="py-3 px-4 text-left">Student ID</th>
                                                <th class="py-3 px-4 text-left">Name</th>
                                                <th class="py-3 px-4 text-left">Status</th>
                                            </tr>
                                        </thead>
                                        <tbody class="text-gray-600">
                                            <?php foreach ($students as $student): 
                                                $default_status = isset($existing_attendance[$student['id']]) ? 
                                                    $existing_attendance[$student['id']] : 'present';
                                            ?>
                                                <tr class="border-b border-gray-200 hover:bg-gray-50">
                                                    <td class="py-3 px-4"><?php echo $student['roll_number']; ?></td>
                                                    <td class="py-3 px-4"><?php echo $student['student_id']; ?></td>
                                                    <td class="py-3 px-4"><?php echo $student['full_name']; ?></td>
                                                    <td class="py-3 px-4">
                                                        <div class="flex space-x-4">
                                                            <label class="inline-flex items-center attendance-option">
                                                                <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="present" 
                                                                    <?php echo ($default_status == 'present') ? 'checked' : ''; ?> 
                                                                    class="form-radio h-4 w-4 text-green-600">
                                                                <span class="ml-2 text-green-600">Present</span>
                                                            </label>
                                                            <label class="inline-flex items-center attendance-option">
                                                                <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="absent" 
                                                                    <?php echo ($default_status == 'absent') ? 'checked' : ''; ?> 
                                                                    class="form-radio h-4 w-4 text-red-600">
                                                                <span class="ml-2 text-red-600">Absent</span>
                                                            </label>
                                                            <label class="inline-flex items-center attendance-option">
                                                                <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="late" 
                                                                    <?php echo ($default_status == 'late') ? 'checked' : ''; ?> 
                                                                    class="form-radio h-4 w-4 text-yellow-600">
                                                                <span class="ml-2 text-yellow-600">Late</span>
                                                            </label>
                                                            <label class="inline-flex items-center attendance-option">
                                                                <input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="half_day" 
                                                                    <?php echo ($default_status == 'half_day') ? 'checked' : ''; ?> 
                                                                    class="form-radio h-4 w-4 text-blue-600">
                                                                <span class="ml-2 text-blue-600">Half Day</span>
                                                            </label>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                                
                                <div class="mt-6 flex justify-center space-x-4">
                                    <button type="button" onclick="selectAllPresent()" class="bg-green-500 hover:bg-green-600 text-white px-4 py-2 ">
                                        <i class="fas fa-check-circle mr-2"></i> Mark All Present
                                    </button>
                                    <button type="button" onclick="selectAllAbsent()" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 ">
                                        <i class="fas fa-times-circle mr-2"></i> Mark All Absent
                                    </button>
                                    <button type="submit" name="submit_attendance" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 ">
                                        <i class="fas fa-save mr-2"></i> Save Attendance
                                    </button>
                                </div>
                            </form>
                        <?php else: ?>
                            <p class="text-center py-4 text-gray-500">
                                No students found in this class and section.
                            </p>
                        <?php endif; ?>
                    </div>
                <?php elseif (count($teacher_classes) == 0 && count($teacher_sections) == 0): ?>
                    <div class="bg-white p-8   text-center">
                        <i class="fas fa-exclamation-triangle text-4xl text-yellow-400 mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-600">You are not assigned to any classes or sections.</h3>
                        <p class="text-gray-500 mt-2">Please contact the administrator to be assigned to a class or section.</p>
                    </div>
                <?php else: ?>
                    <div class="bg-white p-8   text-center">
                        <i class="fas fa-calendar-alt text-4xl text-gray-300 mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-600">Please select a class and section to mark attendance</h3>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Calendar Tab -->
            <div id="calendar-tab" class="tab-content <?php echo $active_tab === 'calendar' ? 'active' : ''; ?>">
                <?php if ($has_access && $class_id > 0 && $section_id > 0): ?>
                    <!-- Calendar Header -->
                    <div class="mb-6 bg-white p-4  ">
                        <div class="flex justify-between items-center mb-4">
                            <h3 class="text-lg font-semibold text-gray-800">
                                <i class="fas fa-calendar-alt mr-2"></i>
                                Attendance Calendar for <?php echo $class_name . " - " . $section_name . " (" . $month_name . ")"; ?>
                            </h3>
                            
                            <div class="flex space-x-2">
                                <a href="attendance.php?class_id=<?php echo $class_id; ?>&section_id=<?php echo $section_id; ?>&month=<?php echo date('Y-m', strtotime($month . ' -1 month')); ?>&tab=calendar" 
                                   class="bg-gray-600 hover:bg-gray-700 text-white px-3 py-1  text-sm">
                                    <i class="fas fa-chevron-left"></i> Prev
                                </a>
                                <a href="attendance.php?class_id=<?php echo $class_id; ?>&section_id=<?php echo $section_id; ?>&month=<?php echo date('Y-m'); ?>&tab=calendar" 
                                   class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-1  text-sm">
                                    Current
                                </a>
                                <a href="attendance.php?class_id=<?php echo $class_id; ?>&section_id=<?php echo $section_id; ?>&month=<?php echo date('Y-m', strtotime($month . ' +1 month')); ?>&tab=calendar" 
                                   class="bg-gray-600 hover:bg-gray-700 text-white px-3 py-1  text-sm">
                                    Next <i class="fas fa-chevron-right"></i>
                                </a>
                            </div>
                        </div>
                        
                        <!-- Attendance Calendar -->
                        <div class="grid grid-cols-7 gap-2 mb-2">
                            <div class="text-center font-medium text-gray-700 p-2">Mon</div>
                            <div class="text-center font-medium text-gray-700 p-2">Tue</div>
                            <div class="text-center font-medium text-gray-700 p-2">Wed</div>
                            <div class="text-center font-medium text-gray-700 p-2">Thu</div>
                            <div class="text-center font-medium text-gray-700 p-2">Fri</div>
                            <div class="text-center font-medium text-gray-700 p-2">Sat</div>
                            <div class="text-center font-medium text-gray-700 p-2">Sun</div>
                        </div>
                        
                        <div class="grid grid-cols-7 gap-2">
                            <?php
                            // Add empty cells for days before the first day of the month
                            for ($i = 1; $i < $first_day; $i++) {
                                echo '<div class="calendar-day border  bg-gray-100 p-2"></div>';
                            }
                            
                            // Add cells for each day of the month
                            for ($day = 1; $day <= $days_in_month; $day++) {
                                $current_date = date('Y-m-d', strtotime($month . '-' . sprintf('%02d', $day)));
                                $is_today = ($current_date == date('Y-m-d')) ? 'border-2 border-blue-500' : 'border';
                                $has_attendance = isset($attendance_summary[$current_date]);
                                $day_of_week = date('N', strtotime($current_date));
                                
                                // Check if it's a weekend (Saturday = 6, Sunday = 7)
                                $is_weekend = ($day_of_week >= 6);
                                
                                if ($has_attendance) {
                                    $summary = $attendance_summary[$current_date];
                                    $attendance_rate = $summary['total_students'] > 0 ? 
                                        round(($summary['present_count'] + ($summary['late_count'] * 0.5) + ($summary['half_day_count'] * 0.5)) / $summary['total_students'] * 100) : 0;
                                    
                                    if ($attendance_rate >= 90) {
                                        $bg_color = 'bg-green-50';
                                        $status_text = 'Good';
                                    } else if ($attendance_rate >= 75) {
                                        $bg_color = 'bg-yellow-50';
                                        $status_text = 'Fair';
                                    } else {
                                        $bg_color = 'bg-red-50';
                                        $status_text = 'Poor';
                                    }
                                } else if ($is_weekend) {
                                    $bg_color = 'bg-gray-100';
                                    $status_text = 'Weekend';
                                } else {
                                    $bg_color = 'bg-gray-50';
                                    $status_text = 'No Data';
                                }
                                
                                echo '<div class="calendar-day border  p-2 ' . $bg_color . ' ' . $is_today . '">';
                                echo '<div class="text-sm font-medium mb-1">' . $day . '</div>';
                                
                                if ($has_attendance) {
                                    echo '<div class="text-xs space-y-1">';
                                    echo '<div class="flex justify-between"><span>Present:</span> <span class="font-medium text-green-600">' . $summary['present_count'] . '</span></div>';
                                    echo '<div class="flex justify-between"><span>Absent:</span> <span class="font-medium text-red-600">' . $summary['absent_count'] . '</span></div>';
                                    if ($summary['late_count'] > 0) {
                                        echo '<div class="flex justify-between"><span>Late:</span> <span class="font-medium text-yellow-600">' . $summary['late_count'] . '</span></div>';
                                    }
                                    if ($summary['half_day_count'] > 0) {
                                        echo '<div class="flex justify-between"><span>Half Day:</span> <span class="font-medium text-blue-600">' . $summary['half_day_count'] . '</span></div>';
                                    }
                                    echo '<div class="border-t pt-1 mt-1 text-center">';
                                    echo '<span class="font-medium">' . $attendance_rate . '%</span>';
                                    echo '</div>';
                                    echo '</div>';
                                    
                                    // Add view details link
                                    echo '<a href="attendance.php?class_id=' . $class_id . '&section_id=' . $section_id . '&month=' . $month . '&date=' . $current_date . '&tab=calendar" 
                                          class="block text-center mt-2 text-xs bg-blue-600 text-white py-1  hover:bg-blue-700">
                                          <i class="fas fa-eye mr-1"></i> Details
                                      </a>';
                                } else {
                                    echo '<div class="text-xs text-gray-500 mt-2">' . $status_text . '</div>';
                                }
                                
                                echo '</div>';
                                
                                // Start new row after Sunday
                                if (($first_day + $day - 1) % 7 == 0 && $day != $days_in_month) {
                                    echo '</div><div class="grid grid-cols-7 gap-2">';
                                }
                            }
                            
                            // Add empty cells for remaining days in the last week
                            $last_day = date('N', strtotime($month . '-' . $days_in_month));
                            if ($last_day < 7) {
                                for ($i = $last_day; $i < 7; $i++) {
                                    echo '<div class="calendar-day border  bg-gray-100 p-2"></div>';
                                }
                            }
                            ?>
                        </div>
                        
                        <!-- Legend -->
                        <div class="mt-4 grid grid-cols-2 md:grid-cols-4 gap-2 text-xs">
                            <div class="flex items-center">
                                <div class="w-4 h-4 bg-green-50 mr-1 border border-green-200"></div>
                                <span>Good (≥90%)</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-4 h-4 bg-yellow-50 mr-1 border border-yellow-200"></div>
                                <span>Fair (75-89%)</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-4 h-4 bg-red-50 mr-1 border border-red-200"></div>
                                <span>Poor (<75%)</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-4 h-4 bg-gray-100 mr-1 border border-gray-200"></div>
                                <span>Weekend</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Detailed Attendance View -->
                    <?php if (!empty($detailed_date) && !empty($detailed_attendance)): ?>
                        <div class="bg-white p-4   mt-6">
                            <h3 class="text-lg font-semibold text-gray-800 mb-4">
                                <i class="fas fa-list mr-2"></i>
                                Detailed Attendance for <?php echo date('d M Y', strtotime($detailed_date)); ?>
                            </h3>
                            
                            <div class="overflow-x-auto">
                                <table class="min-w-full bg-white">
                                    <thead class="bg-gray-200 text-gray-700">
                                        <tr>
                                            <th class="py-3 px-4 text-left">Roll No</th>
                                            <th class="py-3 px-4 text-left">Name</th>
                                            <th class="py-3 px-4 text-left">Status</th>
                                        </tr>
                                    </thead>
                                    <tbody class="text-gray-600">
                                        <?php foreach ($detailed_attendance as $attendance): ?>
                                            <tr class="border-b border-gray-200 hover:bg-gray-50">
                                                <td class="py-3 px-4"><?php echo $attendance['roll_number']; ?></td>
                                                <td class="py-3 px-4"><?php echo $attendance['full_name']; ?></td>
                                                <td class="py-3 px-4">
                                                    <?php 
                                                    $status_class = '';
                                                    switch ($attendance['status']) {
                                                        case 'present':
                                                            $status_class = 'bg-green-100 text-green-800';
                                                            break;
                                                        case 'absent':
                                                            $status_class = 'bg-red-100 text-red-800';
                                                            break;
                                                        case 'late':
                                                            $status_class = 'bg-yellow-100 text-yellow-800';
                                                            break;
                                                        case 'half_day':
                                                            $status_class = 'bg-blue-100 text-blue-800';
                                                            break;
                                                        default:
                                                            $status_class = 'bg-gray-100 text-gray-800';
                                                    }
                                                    ?>
                                                    <span class="px-2 py-1 -full text-xs font-medium <?php echo $status_class; ?>">
                                                        <?php echo ucfirst(str_replace('_', ' ', $attendance['status'])); ?>
                                                    </span>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                <?php elseif (count($teacher_classes) == 0 && count($teacher_sections) == 0): ?>
                    <div class="bg-white p-8   text-center">
                        <i class="fas fa-exclamation-triangle text-4xl text-yellow-400 mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-600">You are not assigned to any classes or sections.</h3>
                        <p class="text-gray-500 mt-2">Please contact the administrator to be assigned to a class or section.</p>
                    </div>
                <?php else: ?>
                    <div class="bg-white p-8   text-center">
                        <i class="fas fa-calendar-alt text-4xl text-gray-300 mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-600">Please select a class and section to view attendance calendar</h3>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        function switchTab(tabName) {
            // Update the hidden field
            document.getElementById('tab_field').value = tabName;
            
            // Hide all tab contents
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Show the selected tab content
            document.getElementById(tabName + '-tab').classList.add('active');
            
            // Update tab button styles
            document.querySelectorAll('.tab-button').forEach(button => {
                button.classList.remove('active');
                button.classList.add('bg-gray-200', 'text-gray-700');
            });
            
            // Activate the clicked tab button
            event.target.classList.remove('bg-gray-200', 'text-gray-700');
            event.target.classList.add('active');
        }
        
        function selectAllPresent() {
            document.querySelectorAll('input[type="radio"][value="present"]').forEach(radio => {
                radio.checked = true;
            });
        }
        
        function selectAllAbsent() {
            document.querySelectorAll('input[type="radio"][value="absent"]').forEach(radio => {
                radio.checked = true;
            });
        }
        
        // Update sections when class changes
        document.getElementById('class_filter').addEventListener('change', function() {
            const classId = this.value;
            const sectionSelect = document.getElementById('section_filter');
            
            if (classId) {
                // Enable section select
                sectionSelect.disabled = false;
                
                // Fetch sections for the selected class
                fetch('get_sections.php?class_id=' + classId)
                    .then(response => response.json())
                    .then(sections => {
                        sectionSelect.innerHTML = '<option value="">Select Section</option>';
                        sections.forEach(section => {
                            sectionSelect.innerHTML += `<option value="${section.id}">${section.section_name}</option>`;
                        });
                    })
                    .catch(error => console.error('Error:', error));
            } else {
                // Disable and reset section select
                sectionSelect.disabled = true;
                sectionSelect.innerHTML = '<option value="">Select Section</option>';
            }
        });
    </script>
</body>
</html>
<?php ob_end_flush(); ?>