<?php
ob_start();
include 'include/header.php'; 
include '../config/conn.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'teacher') {
    header('Location: login.php');
    exit();
}

$teacher_id = $_SESSION['user_id'];
$success = '';
$error = '';

// Get teacher's assigned classes and sections
$teacher_classes = [];
$teacher_sections = [];

// Query to get classes where this teacher is the class teacher
$class_query = "SELECT id, class_name FROM classes WHERE class_teacher_id = ? AND status = 'active'";
$class_stmt = $conn->prepare($class_query);
$class_stmt->bind_param("i", $teacher_id);
$class_stmt->execute();
$class_result = $class_stmt->get_result();

while ($class = $class_result->fetch_assoc()) {
    $teacher_classes[] = $class;
}
$class_stmt->close();

// Query to get sections where this teacher is the section teacher
$section_query = "SELECT s.id, s.section_name, c.class_name, c.id as class_id
                 FROM sections s 
                 JOIN classes c ON s.class_id = c.id 
                 WHERE s.section_teacher_id = ? AND s.status = 'active'";
$section_stmt = $conn->prepare($section_query);
$section_stmt->bind_param("i", $teacher_id);
$section_stmt->execute();
$section_result = $section_stmt->get_result();

while ($section = $section_result->fetch_assoc()) {
    $teacher_sections[] = $section;
}
$section_stmt->close();

// Initialize variables
$class_id = isset($_GET['class_id']) ? intval($_GET['class_id']) : 0;
$section_id = isset($_GET['section_id']) ? intval($_GET['section_id']) : 0;
$exam_id = isset($_GET['exam_id']) ? intval($_GET['exam_id']) : 0;
$student_id = isset($_GET['student_id']) ? intval($_GET['student_id']) : 0;
$action = isset($_GET['action']) ? $_GET['action'] : '';

// Verify that the teacher has access to the selected class/section
$has_access = false;
if ($class_id > 0) {
    foreach ($teacher_classes as $class) {
        if ($class['id'] == $class_id) {
            $has_access = true;
            break;
        }
    }
    
    // If not a class teacher, check if they're a section teacher for any section in this class
    if (!$has_access) {
        foreach ($teacher_sections as $section) {
            if ($section['class_id'] == $class_id) {
                $has_access = true;
                break;
            }
        }
    }
}

// If teacher doesn't have access to the selected class, reset the selection
if ($class_id > 0 && !$has_access) {
    $class_id = 0;
    $section_id = 0;
    $error = "You don't have access to this class.";
}

// Fetch school information
$school_query = "SELECT * FROM school_info LIMIT 1";
$school_result = mysqli_query($conn, $school_query);
$school_info = mysqli_fetch_assoc($school_result);

// If no school info exists, create default
if (!$school_info) {
    $school_info = [
        'school_name' => 'Adino Height School',
        'address' => '123 School Street, City, State',
        'phone' => '+1234567890',
        'email' => 'info@adinoheight.edu',
        'logo' => '',
        'motto' => 'Excellence in Education'
    ];
}

// Fetch all exams
$exams_query = "SELECT id, exam_name, exam_date FROM exams WHERE status = 'active' ORDER BY exam_date DESC";
$exams_result = mysqli_query($conn, $exams_query);

// Fetch sections based on selected class (only those the teacher has access to)
$sections = [];
if ($class_id > 0 && $has_access) {
    $sections_query = "SELECT id, section_name FROM sections WHERE class_id = $class_id AND status = 'active' ORDER BY section_name";
    $sections_result = mysqli_query($conn, $sections_query);
    while ($section = mysqli_fetch_assoc($sections_result)) {
        // Check if teacher has access to this section
        $section_access = false;
        foreach ($teacher_sections as $teacher_section) {
            if ($teacher_section['id'] == $section['id']) {
                $section_access = true;
                break;
            }
        }
        
        // Also check if teacher is the class teacher (class teachers have access to all sections)
        foreach ($teacher_classes as $class) {
            if ($class['id'] == $class_id) {
                $section_access = true;
                break;
            }
        }
        
        if ($section_access) {
            $sections[] = $section;
        }
    }
}

// Fetch students based on selected class and section
$students = [];
if ($class_id > 0 && $section_id > 0 && $has_access) {
    $students_query = "SELECT id, student_id, full_name, roll_number 
                       FROM students 
                       WHERE class = $class_id AND section = $section_id AND status = 'active' 
                       ORDER BY roll_number";
    $students_result = mysqli_query($conn, $students_query);
    while ($student = mysqli_fetch_assoc($students_result)) {
        $students[] = $student;
    }
}

// Use exam name as term (removed automatic calendar year calculation)
$academic_year = date('Y') . '/' . (date('Y') + 1); // Simple academic year format

// Function to get student results data
function getStudentResultsData($conn, $student_id, $exam_id) {
    // Fetch student details
    $student_query = "SELECT s.*, c.class_name, sec.section_name 
                      FROM students s 
                      LEFT JOIN classes c ON s.class = c.id 
                      LEFT JOIN sections sec ON s.section = sec.id 
                      WHERE s.id = $student_id";
    $student_result = mysqli_query($conn, $student_query);
    $student = mysqli_fetch_assoc($student_result);
    
    // Fetch exam details
    $exam_query = "SELECT * FROM exams WHERE id = $exam_id";
    $exam_result = mysqli_query($conn, $exam_query);
    $exam = mysqli_fetch_assoc($exam_result);
    
    // Fetch all results for this student and exam
    $results_query = "SELECT er.*, s.subject_name 
                      FROM exam_results er 
                      JOIN subjects s ON er.subject_id = s.id 
                      WHERE er.student_id = $student_id AND er.exam_id = $exam_id 
                      ORDER BY s.subject_name";
    $results_result = mysqli_query($conn, $results_query);
    $results = [];
    $total_marks = 0;
    $obtained_marks = 0;
    
    while ($row = mysqli_fetch_assoc($results_result)) {
        $results[] = $row;
        $obtained_marks += $row['ca_score'] + $row['exam_score'];
        $total_marks += $exam['total_marks'];
    }
    
    // Calculate overall percentage and grade
    $overall_percentage = $total_marks > 0 ? ($obtained_marks / $total_marks) * 100 : 0;
    
    // Determine overall grade
    $overall_grade = '';
    if ($overall_percentage >= 90) $overall_grade = 'A+';
    elseif ($overall_percentage >= 80) $overall_grade = 'A';
    elseif ($overall_percentage >= 70) $overall_grade = 'B+';
    elseif ($overall_percentage >= 60) $overall_grade = 'B';
    elseif ($overall_percentage >= 50) $overall_grade = 'C';
    elseif ($overall_percentage >= 40) $overall_grade = 'D';
    else $overall_grade = 'F';
    
    // Fetch attendance data for the term
    $attendance_query = "SELECT 
                         COUNT(*) as total_days,
                         SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
                         SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days,
                         SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late_days,
                         SUM(CASE WHEN status = 'half_day' THEN 1 ELSE 0 END) as half_days
                         FROM attendance 
                         WHERE student_id = $student_id 
                         AND YEAR(attendance_date) = YEAR(CURDATE())";
    $attendance_result = mysqli_query($conn, $attendance_query);
    $attendance = $attendance_result ? mysqli_fetch_assoc($attendance_result) : [
        'total_days' => 0,
        'present_days' => 0,
        'absent_days' => 0,
        'late_days' => 0,
        'half_days' => 0
    ];
    
    return [
        'student' => $student,
        'exam' => $exam,
        'results' => $results,
        'total_marks' => $total_marks,
        'obtained_marks' => $obtained_marks,
        'overall_percentage' => $overall_percentage,
        'overall_grade' => $overall_grade,
        'attendance' => $attendance
    ];
}

// Handle PDF generation
if ($action == 'download' && $student_id > 0 && $exam_id > 0 && $has_access) {
    // Clear any previous output
    ob_end_clean();
    
    $data = getStudentResultsData($conn, $student_id, $exam_id);
    extract($data);
    
    // Include TCPDF library
    require_once('../tcpdf/TCPDF-main/tcpdf.php');
    
    // Create new PDF document
    $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
    
    // Set document information
    $pdf->SetCreator('Adino Height School');
    $pdf->SetAuthor('Adino Height School');
    $pdf->SetTitle('Report Card - ' . $student['full_name']);
    $pdf->SetSubject('Student Report Card');
    
    // Get logo path correctly
    $logo_path = '';
    if (!empty($school_info['logo'])) {
        // Check if file exists in both possible locations
        if (file_exists('../uploads/' . $school_info['logo'])) {
            $logo_path = '../uploads/' . $school_info['logo'];
        } elseif (file_exists('uploads/' . $school_info['logo'])) {
            $logo_path = 'uploads/' . $school_info['logo'];
        }
    }
    
    // Set default header data
    $pdf->SetHeaderData('', 0, $school_info['school_name'], 'Academic Report Card');
    
    // Set header and footer fonts
    $pdf->setHeaderFont(Array('helvetica', '', 10));
    $pdf->setFooterFont(Array('helvetica', '', 8));
    
    // Set margins
    $pdf->SetMargins(15, 25, 15);
    $pdf->SetHeaderMargin(10);
    $pdf->SetFooterMargin(10);
    
    // Set auto page breaks
    $pdf->SetAutoPageBreak(TRUE, 25);
    
    // Add a page
    $pdf->AddPage();
    
    // Set font
    $pdf->SetFont('helvetica', '', 10);
    
    // School header with logo
    $html = '
    <div style="text-align: center;">
        <table border="0" cellpadding="0" cellspacing="0" width="100%">
            <tr>
                <td width="20%" style="text-align: left; vertical-align: middle;">';
    
    if (!empty($logo_path) && file_exists($logo_path)) {
        $imageData = base64_encode(file_get_contents($logo_path));
        $imageInfo = getimagesize($logo_path);
        $mimeType = $imageInfo['mime'];
        $html .= '<img src="data:' . $mimeType . ';base64,' . $imageData . '" style="height: 80px; width:80px; object-fit:contain;">';
    }
    
    $html .= '</td>
                <td width="60%" style="text-align: center; vertical-align: middle;">
                    <h1 style="font-size: 18px; margin-bottom: 5px;">' . $school_info['school_name'] . '</h1>
                    <p style="font-size: 12px; margin: 0;">' . $school_info['address'] . '</p>
                    <p style="font-size: 12px; margin: 0;">Phone: ' . $school_info['phone'] . ' | Email: ' . $school_info['email'] . '</p>
                    <p style="font-size: 12px; margin: 0; font-style: italic;">' . $school_info['motto'] . '</p>
                </td>
                <td width="20%" style="text-align: right; vertical-align: middle;">
                    <!-- Empty space for alignment -->
                </td>
            </tr>
        </table>
        <hr style="margin: 10px 0;">
    </div>
    
    <h2 style="text-align: center; font-size: 16px; margin: 10px 0;">STUDENT REPORT CARD</h2>
    
    <table border="0" cellpadding="5" width="100%">
        <tr>
            <td width="50%">
                <strong>Student Name:</strong> ' . $student['full_name'] . '<br>
                <strong>Student ID:</strong> ' . $student['student_id'] . '<br>
                <strong>Roll Number:</strong> ' . $student['roll_number'] . '
            </td>
            <td width="50%">
                <strong>Class:</strong> ' . $student['class_name'] . ' - ' . $student['section_name'] . '<br>
                <strong>Academic Year:</strong> ' . $academic_year . '<br>
                <strong>Term:</strong> ' . $exam['exam_name'] . '
            </td>
        </tr>
        <tr>
            <td>
                <strong>Term Date:</strong> ' . date('F j, Y', strtotime($exam['exam_date'])) . '
            </td>
            <td>
                <strong>Date Issued:</strong> ' . date('F j, Y') . '
            </td>
        </tr>
    </table>
    
    <br>
    
    <h3 style="font-size: 14px; margin: 10px 0;">ACADEMIC PERFORMANCE</h3>
    <table border="1" cellpadding="5" width="100%">
        <tr style="background-color: #f2f2f2;">
            <th width="40%"><strong>Subject</strong></th>
            <th width="15%"><strong>CA Score</strong></th>
            <th width="15%"><strong>Exam Score</strong></th>
            <th width="15%"><strong>Total</strong></th>
            <th width="15%"><strong>Grade</strong></th>
        </tr>';
    
    foreach ($results as $result) {
        $total_score = $result['ca_score'] + $result['exam_score'];
        $html .= '
        <tr>
            <td>' . $result['subject_name'] . '</td>
            <td>' . $result['ca_score'] . '</td>
            <td>' . $result['exam_score'] . '</td>
            <td>' . $total_score . '</td>
            <td>' . $result['grade'] . '</td>
        </tr>';
    }
    
    $html .= '
        <tr style="background-color: #f2f2f2;">
            <td colspan="3" align="right"><strong>Overall Total:</strong></td>
            <td><strong>' . $obtained_marks . '/' . $total_marks . '</strong></td>
            <td><strong>' . $overall_grade . '</strong></td>
        </tr>
        <tr style="background-color: #f2f2f2;">
            <td colspan="3" align="right"><strong>Overall Percentage:</strong></td>
            <td colspan="2"><strong>' . number_format($overall_percentage, 2) . '%</strong></td>
        </tr>
    </table>
    
    <br>
    
    <h3 style="font-size: 14px; margin: 10px 0;">ATTENDANCE RECORD</h3>
    <table border="1" cellpadding="5" width="100%">
        <tr style="background-color: #f2f2f2;">
            <th width="25%"><strong>Total Days</strong></th>
            <th width="25%"><strong>Present</strong></th>
            <th width="25%"><strong>Absent</strong></th>
            <th width="25%"><strong>Attendance %</strong></th>
        </tr>
        <tr>
            <td>' . $attendance['total_days'] . '</td>
            <td>' . $attendance['present_days'] . '</td>
            <td>' . $attendance['absent_days'] . '</td>
            <td>' . ($attendance['total_days'] > 0 ? number_format(($attendance['present_days'] / $attendance['total_days']) * 100, 2) . '%' : '0%') . '</td>
        </tr>
    </table>
    
    <br>
    
    <h3 style="font-size: 14px; margin: 10px 0;">GRADING SYSTEM</h3>
    <table border="1" cellpadding="5" width="100%">
        <tr style="background-color: #f2f2f2;">
            <th width="20%"><strong>Percentage</strong></th>
            <th width="20%"><strong>Grade</strong></th>
            <th width="60%"><strong>Remarks</strong></th>
        </tr>
        <tr>
            <td>90-100%</td>
            <td>A+</td>
            <td>Excellent</td>
        </tr>
        <tr>
            <td>80-89%</td>
            <td>A</td>
            <td>Very Good</td>
        </tr>
        <tr>
            <td>70-79%</td>
            <td>B+</td>
            <td>Good</td>
        </tr>
        <tr>
            <td>60-69%</td>
            <td>B</td>
            <td>Satisfactory</td>
        </tr>
        <tr>
            <td>50-59%</td>
            <td>C</td>
            <td>Average</td>
        </tr>
        <tr>
            <td>40-49%</td>
            <td>D</td>
            <td>Below Average</td>
        </tr>
        <tr>
            <td>Below 40%</td>
            <td>F</td>
            <td>Fail</td>
        </tr>
    </table>
    
    <br>
    
    <table border="0" cellpadding="5" width="100%">
        <tr>
            <td width="50%">
                <strong>Class Teacher\'s Remarks:</strong><br>
                <br><br><br>
                _________________________<br>
                Signature & Date
            </td>
            <td width="50%">
                <strong>Principal\'s Remarks:</strong><br>
                <br><br><br>
                _________________________<br>
                Signature & Date
            </td>
        </tr>
    </table>
    
    <div style="text-align: center; margin-top: 20px; font-size: 10px;">
        This is a computer generated report. No signature is required.
    </div>';
    
    // Output HTML content
    $pdf->writeHTML($html, true, false, true, false, '');
    
    // Close and output PDF document
    $pdf->Output('report_card_' . $student['student_id'] . '_' . $exam['exam_name'] . '.pdf', 'D');
    exit();
}

// Handle view action
if ($action == 'view' && $student_id > 0 && $exam_id > 0 && $has_access) {
    $data = getStudentResultsData($conn, $student_id, $exam_id);
    extract($data);
    
    // Get logo path correctly
    $logo_path = '';
    if (!empty($school_info['logo'])) {
        // Check if file exists in both possible locations
        if (file_exists('../uploads/' . $school_info['logo'])) {
            $logo_path = '../uploads/' . $school_info['logo'];
        } elseif (file_exists('uploads/' . $school_info['logo'])) {
            $logo_path = 'uploads/' . $school_info['logo'];
        }
    }
    
    // Display the report in HTML format
    echo '<!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Student Report - ' . $student['full_name'] . '</title>
        <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
        <style>
            @media print {
                .no-print {
                    display: none !important;
                }
                body {
                    margin: 0;
                    padding: 15px;
                }
            }
        </style>
    </head>
    <body class="bg-gray-100">
        <div class="container mx-auto p-4 bg-white ">
            <div class="text-center mb-6 no-print">
            
                <a href="report.php?class_id=' . $class_id . '&section_id=' . $section_id . '&exam_id=' . $exam_id . '" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg ml-2">
                    <i class="fas fa-arrow-left mr-2"></i> Back to List
                </a>
            </div>
            
            <div style="text-align: center;">
                <table border="0" cellpadding="0" cellspacing="0" width="100%">
                    <tr>
                        <td width="20%" style="text-align: left; vertical-align: middle;">';
    
    if (!empty($logo_path) && file_exists($logo_path)) {
        echo '<img src="' . $logo_path . '" style="height: 60px;">';
    }
    
    echo '</td>
                        <td width="60%" style="text-align: center; vertical-align: middle;">
                            <h1 style="font-size: 18px; margin-bottom: 5px;">' . $school_info['school_name'] . '</h1>
                            <p style="font-size: 12px; margin: 0;">' . $school_info['address'] . '</p>
                            <p style="font-size: 12px; margin: 0;">Phone: ' . $school_info['phone'] . ' | Email: ' . $school_info['email'] . '</p>
                            <p style="font-size: 12px; margin: 0; font-style: italic;">' . $school_info['motto'] . '</p>
                        </td>
                        <td width="20%" style="text-align: right; vertical-align: middle;">
                            <!-- Empty space for alignment -->
                        </td>
                    </tr>
                </table>
                <hr style="margin: 10px 0;">
            </div>
            
            <h2 style="text-align: center; font-size: 16px; margin: 10px 0;">STUDENT REPORT CARD</h2>
            
            <table border="0" cellpadding="5" width="100%">
                <tr>
                    <td width="50%">
                        <strong>Student Name:</strong> ' . $student['full_name'] . '<br>
                        <strong>Student ID:</strong> ' . $student['student_id'] . '<br>
                        <strong>Roll Number:</strong> ' . $student['roll_number'] . '
                    </td>
                    <td width="50%">
                        <strong>Class:</strong> ' . $student['class_name'] . ' - ' . $student['section_name'] . '<br>
                        <strong>Academic Year:</strong> ' . $academic_year . '<br>
                        <strong>Term:</strong> ' . $exam['exam_name'] . '
                    </td>
                </tr>
                <tr>
                    <td>
                        <strong>Term Date:</strong> ' . date('F j, Y', strtotime($exam['exam_date'])) . '
                    </td>
                    <td>
                        <strong>Date Issued:</strong> ' . date('F j, Y') . '
                    </td>
                </tr>
            </table>
            
            <br>
            
            <h3 style="font-size: 14px; margin: 10px 0;">ACADEMIC PERFORMANCE</h3>
            <table border="1" cellpadding="5" width="100%" style="border-collapse: collapse;">
                <tr style="background-color: #f2f2f2;">
                    <th width="40%"><strong>Subject</strong></th>
                    <th width="15%"><strong>CA Score</strong></th>
                    <th width="15%"><strong>Exam Score</strong></th>
                    <th width="15%"><strong>Total</strong></th>
                    <th width="15%"><strong>Grade</strong></th>
                </tr>';
    
    foreach ($results as $result) {
        $total_score = $result['ca_score'] + $result['exam_score'];
        echo '
                <tr>
                    <td>' . $result['subject_name'] . '</td>
                    <td>' . $result['ca_score'] . '</td>
                    <td>' . $result['exam_score'] . '</td>
                    <td>' . $total_score . '</td>
                    <td>' . $result['grade'] . '</td>
                </tr>';
    }
    
    echo '
                <tr style="background-color: #f2f2f2;">
                    <td colspan="3" align="right"><strong>Overall Total:</strong></td>
                    <td><strong>' . $obtained_marks . '/' . $total_marks . '</strong></td>
                    <td><strong>' . $overall_grade . '</strong></td>
                </tr>
                <tr style="background-color: #f2f2f2;">
                    <td colspan="3" align="right"><strong>Overall Percentage:</strong></td>
                    <td colspan="2"><strong>' . number_format($overall_percentage, 2) . '%</strong></td>
                </tr>
            </table>
            
            <br>
            
            <h3 style="font-size: 14px; margin: 10px 0;">ATTENDANCE RECORD</h3>
            <table border="1" cellpadding="5" width="100%" style="border-collapse: collapse;">
                <tr style="background-color: #f2f2f2;">
                    <th width="25%"><strong>Total Days</strong></th>
                    <th width="25%"><strong>Present</strong></th>
                    <th width="25%"><strong>Absent</strong></th>
                    <th width="25%"><strong>Attendance %</strong></th>
                </tr>
                <tr>
                    <td>' . $attendance['total_days'] . '</td>
                    <td>' . $attendance['present_days'] . '</td>
                    <td>' . $attendance['absent_days'] . '</td>
                    <td>' . ($attendance['total_days'] > 0 ? number_format(($attendance['present_days'] / $attendance['total_days']) * 100, 2) . '%' : '0%') . '</td>
                </tr>
            </table>
            
            <br>
            
            <h3 style="font-size: 14px; margin: 10px 0;">GRADING SYSTEM</h3>
            <table border="1" cellpadding="5" width="100%" style="border-collapse: collapse;">
                <tr style="background-color: #f2f2f2;">
                    <th width="20%"><strong>Percentage</strong></th>
                    <th width="20%"><strong>Grade</strong></th>
                    <th width="60%"><strong>Remarks</strong></th>
                </tr>
                <tr>
                    <td>90-100%</td>
                    <td>A+</td>
                    <td>Excellent</td>
                </tr>
                <tr>
                    <td>80-89%</td>
                    <td>A</td>
                    <td>Very Good</td>
                </tr>
                <tr>
                    <td>70-79%</td>
                    <td>B+</td>
                    <td>Good</td>
                </tr>
                <tr>
                    <td>60-69%</td>
                    <td>B</td>
                    <td>Satisfactory</td>
                </tr>
                <tr>
                    <td>50-59%</td>
                    <td>C</td>
                    <td>Average</td>
                </tr>
                <tr>
                    <td>40-49%</td>
                    <td>D</td>
                    <td>Below Average</td>
                </tr>
                <tr>
                    <td>Below 40%</td>
                    <td>F</td>
                    <td>Fail</td>
                </tr>
            </table>
            
            <br>
            
            <table border="0" cellpadding="5" width="100%">
                <tr>
                    <td width="50%">
                        <strong>Class Teacher\'s Remarks:</strong><br>
                        <br><br><br>
                        _________________________<br>
                        Signature & Date
                    </td>
                    <td width="50%">
                        <strong>Principal\'s Remarks:</strong><br>
                        <br><br><br>
                        _________________________<br>
                        Signature & Date
                    </td>
                </tr>
            </table>
            
            <div style="text-align: center; margin-top: 20px; font-size: 10px;">
                This is a computer generated report. No signature is required.
            </div>
        </div>
    </body>
    </html>';
    exit();
}

// Get class, section, and exam names for display
$class_name = '';
$section_name = '';
$exam_name = '';

if ($class_id > 0) {
    $class_name_query = "SELECT class_name FROM classes WHERE id = $class_id";
    $class_name_result = mysqli_query($conn, $class_name_query);
    if ($class_name_result && mysqli_num_rows($class_name_result) > 0) {
        $class_name = mysqli_fetch_assoc($class_name_result)['class_name'];
    }
}

if ($section_id > 0) {
    $section_name_query = "SELECT section_name FROM sections WHERE id = $section_id";
    $section_name_result = mysqli_query($conn, $section_name_query);
    if ($section_name_result && mysqli_num_rows($section_name_result) > 0) {
        $section_name = mysqli_fetch_assoc($section_name_result)['section_name'];
    }
}

if ($exam_id > 0) {
    $exam_name_query = "SELECT exam_name FROM exams WHERE id = $exam_id";
    $exam_name_result = mysqli_query($conn, $exam_name_query);
    if ($exam_name_result && mysqli_num_rows($exam_name_result) > 0) {
        $exam_name = mysqli_fetch_assoc($exam_name_result)['exam_name'];
    }
}
?>


            
                <div class="bg-white p-6 ">
                    <div class="flex justify-between items-center mb-6">
                        <h2 class="text-2xl font-bold text-gray-800">Student Results & Report Cards</h2>
                    </div>

                    <?php if (!empty($success)): ?>
                        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                            <?php echo $success; ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($error)): ?>
                        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                            <?php echo $error; ?>
                        </div>
                    <?php endif; ?>

                    <!-- Filter Section -->
                    <div class="mb-6 bg-gray-50 p-4 rounded-lg">
                        <form method="GET" action="" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                            <!-- Class Filter -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Select Class</label>
                                <select name="class_id" id="class_filter" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                    <option value="">Select Class</option>
                                    <?php foreach ($teacher_classes as $class): ?>
                                        <option value="<?php echo $class['id']; ?>" <?php echo ($class_id == $class['id']) ? 'selected' : ''; ?>>
                                            <?php echo $class['class_name']; ?> (Class Teacher)
                                        </option>
                                    <?php endforeach; ?>
                                    <?php 
                                    // Add sections where teacher is section teacher (but not already listed as class teacher)
                                    $added_classes = array_column($teacher_classes, 'id');
                                    foreach ($teacher_sections as $section): 
                                        if (!in_array($section['class_id'], $added_classes)):
                                    ?>
                                        <option value="<?php echo $section['class_id']; ?>" <?php echo ($class_id == $section['class_id']) ? 'selected' : ''; ?>>
                                            <?php echo $section['class_name']; ?> (Section Teacher)
                                        </option>
                                    <?php 
                                        endif;
                                    endforeach; 
                                    ?>
                                </select>
                            </div>
                            
                            <!-- Section Filter -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Select Section</label>
                                <select name="section_id" id="section_filter" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" <?php echo ($class_id == 0) ? 'disabled' : ''; ?> required>
                                    <option value="">Select Section</option>
                                    <?php foreach ($sections as $section): ?>
                                        <option value="<?php echo $section['id']; ?>" <?php echo ($section_id == $section['id']) ? 'selected' : ''; ?>>
                                            <?php echo $section['section_name']; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <!-- Exam Filter -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Select Exam</label>
                                <select name="exam_id" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                    <option value="">Select Exam</option>
                                    <?php 
                                    // Reset pointer and loop through exams again
                                    mysqli_data_seek($exams_result, 0);
                                    while ($exam = mysqli_fetch_assoc($exams_result)): 
                                    ?>
                                        <option value="<?php echo $exam['id']; ?>" <?php echo ($exam_id == $exam['id']) ? 'selected' : ''; ?>>
                                            <?php echo $exam['exam_name'] . ' (' . date('M j, Y', strtotime($exam['exam_date'])) . ')'; ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            
                            <!-- Buttons -->
                            <div class="flex items-end space-x-2">
                                <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg flex items-center h-10">
                                    <i class="fas fa-filter mr-2"></i> Load Students
                                </button>
                                <a href="report.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg flex items-center h-10">
                                    <i class="fas fa-times mr-2"></i> Clear
                                </a>
                            </div>
                        </form>
                    </div>

                    <?php if ($class_id > 0 && $section_id > 0 && $exam_id > 0 && $has_access): ?>
                        <!-- Students List -->
                        <div class="mb-6 bg-white p-4 ">
                            <div class="flex justify-between items-center mb-4">
                                <h3 class="text-lg font-semibold text-gray-800">
                                    Students in <?php echo $class_name . " - " . $section_name . " - " . $exam_name; ?>
                                </h3>
                            </div>
                            
                            <?php if (!empty($students)): ?>
                                <div class="overflow-x-auto">
                                    <table class="min-w-full bg-white">
                                        <thead class="bg-gray-200 text-gray-700">
                                            <tr>
                                                <th class="py-3 px-4 text-left">Roll No</th>
                                                <th class="py-3 px-4 text-left">Student ID</th>
                                                <th class="py-3 px-4 text-left">Name</th>
                                                <th class="py-3 px-4 text-left">Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody class="text-gray-600">
                                            <?php foreach ($students as $student): ?>
                                                <tr class="border-b border-gray-200 hover:bg-gray-50">
                                                    <td class="py-3 px-4"><?php echo $student['roll_number']; ?></td>
                                                    <td class="py-3 px-4"><?php echo $student['student_id']; ?></td>
                                                    <td class="py-3 px-4"><?php echo $student['full_name']; ?></td>
                                                    <td class="py-3 px-4">
                                                        <div class="flex space-x-2">
                                                            <a href="report.php?class_id=<?php echo $class_id; ?>&section_id=<?php echo $section_id; ?>&exam_id=<?php echo $exam_id; ?>&student_id=<?php echo $student['id']; ?>&action=view" 
                                                               class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 rounded flex items-center">
                                                                <i class="fas fa-eye mr-1"></i> View
                                                            </a>
                                                          
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php else: ?>
                                <p class="text-center py-4 text-gray-500">
                                    No students found in this class and section.
                                </p>
                            <?php endif; ?>
                        </div>
                    <?php elseif ($class_id > 0 && !$has_access): ?>
                        <div class="bg-white p-8  text-center">
                            <i class="fas fa-exclamation-triangle text-4xl text-yellow-500 mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-600">You don't have permission to access this class/section</h3>
                        </div>
                    <?php else: ?>
                        <div class="bg-white p-8  text-center">
                            <i class="fas fa-file-pdf text-4xl text-gray-300 mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-600">Please select class, section, and exam to view student results</h3>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <script>
    // JavaScript to handle dynamic section loading based on class selection
    document.getElementById('class_filter').addEventListener('change', function() {
        const classId = this.value;
        const sectionSelect = document.getElementById('section_filter');
        
        if (classId) {
            // Enable section dropdown
            sectionSelect.disabled = false;
            
            // Fetch sections for the selected class via AJAX
            fetch('get_sections.php?class_id=' + classId + '&teacher_id=<?php echo $teacher_id; ?>')
                .then(response => response.json())
                .then(sections => {
                    // Clear existing options except the first one
                    while (sectionSelect.options.length > 1) {
                        sectionSelect.remove(1);
                    }
                    
                    // Add new options
                    sections.forEach(section => {
                        const option = document.createElement('option');
                        option.value = section.id;
                        option.textContent = section.section_name;
                        sectionSelect.appendChild(option);
                    });
                })
                .catch(error => console.error('Error fetching sections:', error));
        } else {
            // Disable and clear section dropdown if no class is selected
            sectionSelect.disabled = true;
            while (sectionSelect.options.length > 1) {
                sectionSelect.remove(1);
            }
        }
    });
    </script>
</body>
</html>

<?php
ob_end_flush();
?>