<?php
ob_start();
include 'include/header.php';
include '../config/conn.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
    header('Location: ../portal_login.php');
    exit();
}

$teacher_id = $_SESSION['user_id'];

// Fetch teacher data
$sql = "SELECT * FROM users WHERE id = ? AND role = 'teacher'";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // Teacher not found
    header('Location: ../portal_login.php');
    exit();
}

$teacher = $result->fetch_assoc();
$stmt->close();

// Set default date range (current month)
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');

// Fetch attendance records for the teacher
$attendance_sql = "SELECT * FROM teacher_attendance 
                   WHERE teacher_id = ? 
                   AND attendance_date BETWEEN ? AND ?
                   ORDER BY attendance_date DESC";
$stmt = $conn->prepare($attendance_sql);
$stmt->bind_param("iss", $teacher_id, $start_date, $end_date);
$stmt->execute();
$attendance_result = $stmt->get_result();
$attendance_records = [];

while ($row = $attendance_result->fetch_assoc()) {
    $attendance_records[$row['attendance_date']] = $row;
}
$stmt->close();

// Calculate attendance statistics
$stats_sql = "SELECT 
    COUNT(*) as total_days,
    SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_days,
    SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_days,
    SUM(CASE WHEN status = 'late' THEN 1 ELSE 0 END) as late_days,
    SUM(CASE WHEN status = 'leave' THEN 1 ELSE 0 END) as leave_days
    FROM teacher_attendance 
    WHERE teacher_id = ? 
    AND attendance_date BETWEEN ? AND ?";
$stmt = $conn->prepare($stats_sql);
$stmt->bind_param("iss", $teacher_id, $start_date, $end_date);
$stmt->execute();
$stats_result = $stmt->get_result();
$stats = $stats_result->fetch_assoc();
$stmt->close();

// Calculate attendance percentage
$attendance_percentage = $stats['total_days'] > 0 
    ? round(($stats['present_days'] + $stats['late_days']) / $stats['total_days'] * 100, 2) 
    : 0;

ob_end_flush();
?>


    <style>
        .attendance-status {
            padding: 6px 12px;
          
            font-size: 14px;
            font-weight: 500;
            display: inline-block;
        }
        .status-present {
            background-color: #DCFCE7;
            color: #16A34A;
        }
        .status-absent {
            background-color: #FEE2E2;
            color: #DC2626;
        }
        .status-late {
            background-color: #FEF3C7;
            color: #D97706;
        }
        .status-leave {
            background-color: #EDE9FE;
            color: #7C3AED;
        }
        .calendar-day {
            height: 80px;
            border: 1px solid #e5e7eb;
            padding: 5px;
            position: relative;
            cursor: pointer;
        }
        .calendar-day.today {
            border: 2px solid #3b82f6;
        }
        .calendar-day.has-attendance:hover {
            background-color: #f0f9ff;
        }
        .calendar-day.present {
            background-color: #DCFCE7;
        }
        .calendar-day.absent {
            background-color: #FEE2E2;
        }
        .calendar-day.late {
            background-color: #FEF3C7;
        }
        .calendar-day.leave {
            background-color: #EDE9FE;
        }
        .progress-bar {
            height: 8px;
            border-radius: 4px;
            overflow: hidden;
            background-color: #e5e7eb;
        }
        .progress-fill {
            height: 100%;
            border-radius: 4px;
            transition: width 0.5s ease-in-out;
        }
       
    </style>
</head>
<body class="bg-gray-100">
<div class="container mx-auto px-2 py-2">
    <div class="bg-white p-6 ">
        <div class="flex justify-between items-center mb-6">
            <div>
                <h2 class="text-2xl font-bold text-gray-800">My Attendance</h2>
                <p class="text-gray-600">View your attendance records and statistics</p>
            </div>
            <div class="text-sm text-gray-600 text-right">
                <div class="font-semibold"><?php echo $teacher['full_name']; ?></div>
                <div>ID: <?php echo $teacher['teacher_id']; ?></div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 mb-6">
            <div class="stat-card present">
                <div class="text-2xl font-bold "><?php echo $stats['present_days']; ?></div>
                <div class="text-sm text-gray-600">Present Days</div>
            </div>
            <div class="stat-card absent">
                <div class="text-2xl font-bold "><?php echo $stats['absent_days']; ?></div>
                <div class="text-sm text-gray-600">Absent Days</div>
            </div>
            <div class="stat-card late">
                <div class="text-2xl font-bold "><?php echo $stats['late_days']; ?></div>
                <div class="text-sm text-gray-600">Late Days</div>
            </div>
            <div class="stat-card leave">
                <div class="text-2xl font-bold "><?php echo $stats['leave_days']; ?></div>
                <div class="text-sm text-gray-600">Leave Days</div>
            </div>
            <div class="stat-card">
                <div class="text-2xl font-bold "><?php echo $attendance_percentage; ?>%</div>
                <div class="text-sm text-gray-600">Attendance Rate</div>
                <div class="progress-bar mt-2">
                    <div class="progress-fill bg-blue-600" style="width: <?php echo $attendance_percentage; ?>%"></div>
                </div>
            </div>
        </div>

        <!-- Filter Section -->
        <div class="mb-6 bg-gray-50 p-4 ">
            <h3 class="text-lg font-semibold text-gray-800 mb-3">Filter Attendance Records</h3>
            <form method="GET" action="" class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Start Date</label>
                    <input type="date" name="start_date" value="<?php echo $start_date; ?>" 
                        class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">End Date</label>
                    <input type="date" name="end_date" value="<?php echo $end_date; ?>" 
                        class="w-full px-4 py-2 border  focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                <div class="flex items-end space-x-2">
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 ">
                        <i class="fas fa-filter mr-2"></i> Apply Filter
                    </button>
                    <a href="teacher_my_attendance.php" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 ">
                        <i class="fas fa-times mr-2"></i> Clear
                    </a>
                </div>
            </form>
        </div>

        <!-- Attendance Calendar View -->
        <div class="mb-6 bg-white p-4  ">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Attendance Calendar</h3>
            
            <?php
            // Generate calendar for the selected month
            $current_month = date('Y-m', strtotime($start_date));
            $month_name = date('F Y', strtotime($current_month));
            $first_day = date('N', strtotime($current_month . '-01'));
            $days_in_month = date('t', strtotime($current_month . '-01'));
            ?>
            
            <div class="mb-4 flex justify-between items-center">
                <h4 class="text-md font-medium text-gray-700"><?php echo $month_name; ?></h4>
            </div>
            
            <div class="grid grid-cols-7 gap-2 mb-2">
                <div class="text-center font-medium text-gray-700">Mon</div>
                <div class="text-center font-medium text-gray-700">Tue</div>
                <div class="text-center font-medium text-gray-700">Wed</div>
                <div class="text-center font-medium text-gray-700">Thu</div>
                <div class="text-center font-medium text-gray-700">Fri</div>
                <div class="text-center font-medium text-gray-700">Sat</div>
                <div class="text-center font-medium text-gray-700">Sun</div>
            </div>
            
            <div class="grid grid-cols-7 gap-2">
                <?php
                // Add empty cells for days before the first day of the month
                for ($i = 1; $i < $first_day; $i++) {
                    echo '<div class="calendar-day bg-gray-100"></div>';
                }
                
                // Add cells for each day of the month
                for ($day = 1; $day <= $days_in_month; $day++) {
                    $current_date = date('Y-m-d', strtotime($current_month . '-' . sprintf('%02d', $day)));
                    $is_today = ($current_date == date('Y-m-d')) ? 'today' : '';
                    $has_attendance = isset($attendance_records[$current_date]);
                    $status_class = $has_attendance ? $attendance_records[$current_date]['status'] : '';
                    
                    echo '<div class="calendar-day ' . $is_today . ' ' . $status_class . '">';
                    echo '<div class="text-sm font-medium">' . $day . '</div>';
                    
                    if ($has_attendance) {
                        $status = $attendance_records[$current_date]['status'];
                        $time = date('H:i', strtotime($attendance_records[$current_date]['recorded_at']));
                        
                        echo '<div class="text-xs mt-1">';
                        echo '<span class="attendance-status status-'.$status.'">' . ucfirst($status) . '</span>';
                        echo '</div>';
                        
                        if ($status === 'present' || $status === 'late') {
                            echo '<div class="text-xs text-gray-500 mt-1">' . $time . '</div>';
                        }
                    } else {
                        // Check if it's a future date
                        if (strtotime($current_date) > time()) {
                            echo '<div class="text-xs text-gray-400 mt-2">Future</div>';
                        } else {
                            echo '<div class="text-xs text-gray-400 mt-2">No record</div>';
                        }
                    }
                    
                    echo '</div>';
                    
                    // Start new row after Sunday
                    if (($first_day + $day - 1) % 7 == 0 && $day != $days_in_month) {
                        echo '</div><div class="grid grid-cols-7 gap-2">';
                    }
                }
                
                // Add empty cells for remaining days in the last week
                $last_day = date('N', strtotime($current_month . '-' . $days_in_month));
                if ($last_day < 7) {
                    for ($i = $last_day; $i < 7; $i++) {
                        echo '<div class="calendar-day bg-gray-100"></div>';
                    }
                }
                ?>
            </div>
            
            <div class="mt-4 flex flex-wrap gap-4 text-xs">
                <div class="flex items-center">
                    <div class="w-4 h-4 bg-green-100 mr-1 border border-green-300"></div>
                    <span>Present</span>
                </div>
                <div class="flex items-center">
                    <div class="w-4 h-4 bg-red-100 mr-1 border border-red-300"></div>
                    <span>Absent</span>
                </div>
                <div class="flex items-center">
                    <div class="w-4 h-4 bg-yellow-100 mr-1 border border-yellow-300"></div>
                    <span>Late</span>
                </div>
                <div class="flex items-center">
                    <div class="w-4 h-4 bg-purple-100 mr-1 border border-purple-300"></div>
                    <span>Leave</span>
                </div>
                <div class="flex items-center">
                    <div class="w-4 h-4 bg-blue-100 mr-1 border border-blue-300"></div>
                    <span>Today</span>
                </div>
            </div>
        </div>

        <!-- Detailed Attendance Records -->
        <div class="bg-white p-4  ">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Detailed Attendance Records</h3>
            
            <?php if (count($attendance_records) > 0): ?>
                <div class="overflow-x-auto">
                    <table class="min-w-full bg-white">
                        <thead class="bg-gray-200 text-gray-700">
                            <tr>
                                <th class="py-3 px-4 text-left">Date</th>
                                <th class="py-3 px-4 text-left">Day</th>
                                <th class="py-3 px-4 text-left">Status</th>
                                <th class="py-3 px-4 text-left">Time Recorded</th>
                                
                            </tr>
                        </thead>
                        <tbody class="text-gray-600">
                            <?php 
                            // Sort records by date descending
                            krsort($attendance_records);
                            
                            foreach ($attendance_records as $date => $record): 
                                $day_name = date('l', strtotime($date));
                                $is_weekend = (date('N', strtotime($date)) >= 6);
                            ?>
                                <tr class="border-b border-gray-200 hover:bg-gray-50 <?php echo $is_weekend ? 'bg-gray-50' : ''; ?>">
                                    <td class="py-3 px-4"><?php echo date('M j, Y', strtotime($date)); ?></td>
                                    <td class="py-3 px-4"><?php echo $day_name; ?></td>
                                    <td class="py-3 px-4">
                                        <span class="attendance-status status-<?php echo $record['status']; ?>">
                                            <?php echo ucfirst($record['status']); ?>
                                        </span>
                                    </td>
                                    <td class="py-3 px-4">
                                        <?php echo date('H:i:s', strtotime($record['recorded_at'])); ?>
                                    </td>
                              
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p class="text-center py-8 text-gray-500">
                    No attendance records found for the selected date range.
                </p>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
// Function to print attendance report
function printReport() {
    window.print();
}

// Update calendar day titles with full date on hover
document.addEventListener('DOMContentLoaded', function() {
    const calendarDays = document.querySelectorAll('.calendar-day');
    
    calendarDays.forEach(day => {
        const dayNumber = day.querySelector('.text-sm')?.textContent;
        if (dayNumber) {
            const month = '<?php echo date("F", strtotime($current_month)); ?>';
            const year = '<?php echo date("Y", strtotime($current_month)); ?>';
            const fullDate = `${month} ${dayNumber}, ${year}`;
            day.setAttribute('title', fullDate);
        }
    });
});
</script>

<?php
include 'include/footer.php';
$conn->close();
?>
</body>
</html>